% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_correction_factors.R
\name{get_correction_factors}
\alias{get_correction_factors}
\title{Correction Factors for Tree Plot Areas Intersected by Stand Boundaries}
\usage{
get_correction_factors(
  angle_counts,
  boundaries,
  verbose = TRUE,
  stop_on_error = FALSE,
  skip_check = FALSE,
  counting_factor = 4
)
}
\arguments{
\item{angle_counts}{A \code{\link{data.frame}} containing angle counts.
It has to have columns named by the contents of
either\cr
\code{\link{get_defaults}()[["angle_counts"]]} or \cr
\code{getOption("treePlotArea")[["angle_counts"]]}.\cr
Could be
\code{bw2bwi2022de(get(data("trees", package = "treePlotArea")))}).}

\item{boundaries}{A \code{\link{data.frame}} containing boundaries.
It has to have columns named by the contents of
either\cr
\code{\link{get_defaults}()[["boundaries"]]} or \cr
\code{getOption("treePlotArea")[["boundaries"]]}.\cr
Could be
\code{get(data("boundaries", package = "treePlotArea"))} or the
output of
\code{\link{get_boundary_polygons}}.}

\item{verbose}{Be verbose?}

\item{stop_on_error}{Passed to \code{\link{get_boundary_polygons}}.}

\item{skip_check}{We usually check if the angle counts are
suitable
(for example whether a diameter at breast height, a horizontal distance and
an azimuth
measurement are given). Skip this check? This might be of interest if you
want to check whether another plot with no dbh recorded (for example a
corner) is intersected by a boundary.}

\item{counting_factor}{The basal area factor used in counting the trees. For
tally trees in the German national forest inventory its value is 4 [m^2].}
}
\value{
A  \code{\link{data.frame}} containing the correction factors and a
status giving information on possibly errors.
}
\description{
Get correction factors for an angle count table (i.e. a
\code{\link{data.frame}}) and a corresponding boundary
table (i.e. a \code{\link{data.frame}}).
}
\details{
The columns in the names have to be named according to the values of
\code{getOption("treePlotArea")}.
If they do not: you can either rename the columns
or set the option accordingly, probably using \code{\link{set_options}}.
}
\examples{
data("trees", "boundaries", package = "treePlotArea")

# For CRAN's sake: draw a subset
tracts <-  c(sample(boundaries[["tnr"]], 20), 10056)

# Calculate correction factors
trees <- subset(trees, tnr \%in\% tracts)
boundaries <- subset(boundaries, tnr \%in\% tracts)
angle_counts <- bw2bwi2022de(trees)
validate_data(x = boundaries)
validate_data(x = angle_counts)
boundary_polygons <- get_boundary_polygons(boundaries)
correction_factors <- get_correction_factors(angle_counts, boundary_polygons)
summary(correction_factors$status)

#  Select valid angle count trees only
valid_angle_counts <- select_valid_angle_count_trees(angle_counts)
correction_factors <- get_correction_factors(valid_angle_counts,
                                             boundary_polygons)
summary(correction_factors$status)

# Select a single tree
tnr <- 10056
enr <- 4
bnr <- 3
tree <- valid_angle_counts[valid_angle_counts[["tnr"]] == tnr &
                     valid_angle_counts[["enr"]] == enr &
                     valid_angle_counts[["bnr"]] == bnr, TRUE]
bounds <- boundaries[boundaries[["tnr"]] == tnr & boundaries[["enr"]] == enr,
                     TRUE]
get_correction_factors(tree, bounds)

# Dead wood plots:
dead_wood_plots <- unique(trees[TRUE, c("tnr", "enr")])
dead_wood_plots[["bnr"]] <- 0
dead_wood_plots[["hori"]] <- 0
dead_wood_plots[["azi"]] <- 0
dead_wood_plots[["bhd"]] <- 200
get_correction_factors(dead_wood_plots, boundary_polygons,
                       skip_check = TRUE)
# Set the deadwood plot's radius to 500 mm
dead_wood_plots[["bhd"]] <- 5000
# The counting factor has unit square meters per area.
# Area is hardcoded to 10000 [square meters], so to get a plot radius that's
# equal to the dbh, we need 2 * sqrt(counting_factor) / sqrt(10000) to be
# equal to 1.
get_correction_factors(dead_wood_plots, boundary_polygons,
                       skip_check = TRUE,
                       counting_factor = 2500)
}
\seealso{
set_options
}
