% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xmu_make_top_twin_models.R
\name{xmu_make_TwinSuperModel}
\alias{xmu_make_TwinSuperModel}
\title{Helper to make a basic top, MZ, and DZ model.}
\usage{
xmu_make_TwinSuperModel(
  name = "twin_super",
  mzData,
  dzData,
  selDVs,
  selCovs = NULL,
  sep = NULL,
  type = c("Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS"),
  allContinuousMethod = c("cumulants", "marginals"),
  numObsMZ = NULL,
  numObsDZ = NULL,
  nSib = 2,
  equateMeans = TRUE,
  weightVar = NULL,
  bVector = FALSE,
  dropMissingDef = TRUE,
  verbose = FALSE
)
}
\arguments{
\item{name}{for the supermodel}

\item{mzData}{Dataframe containing the MZ data}

\item{dzData}{Dataframe containing the DZ data}

\item{selDVs}{List of manifest base names (e.g. BMI, NOT 'BMI_T1') (OR, you don't set "sep", the full variable names)}

\item{selCovs}{List of covariate base names (e.g. age, NOT 'age_T1') (OR, you don't set "sep", the full variable names)}

\item{sep}{string used to expand selDVs into selVars, i.e., "_T" to expand BMI into BMI_T1 and BMI_T2 (optional but STRONGLY encouraged)}

\item{type}{One of 'Auto','FIML','cov', 'cor', 'WLS','DWLS', or 'ULS'. Auto tries to react to the incoming mxData type (raw/cov).}

\item{allContinuousMethod}{"cumulants" or "marginals". Used in all-continuous WLS data to determine if a means model needed.}

\item{numObsMZ}{Number of MZ observations contributing (for summary data only)}

\item{numObsDZ}{Number of DZ observations contributing (for summary data only)}

\item{nSib}{Number of members per family (default = 2)}

\item{equateMeans}{Whether to equate T1 and T2 means (default = TRUE).}

\item{weightVar}{If provided, a vector objective will be used to weight the data. (default = NULL).}

\item{bVector}{Whether to compute row-wise likelihoods (defaults to FALSE).}

\item{dropMissingDef}{Whether to automatically drop missing def var rows for the user (default = TRUE). You get a polite note.}

\item{verbose}{(default = FALSE)}
}
\value{
\itemize{
\item \code{\link[=mxModel]{mxModel()}}s for top, MZ and DZ.
}
}
\description{
\code{xmu_make_TwinSuperModel} makes basic twin model containing \code{top}, \code{MZ}, and \code{DZ} models. It intelligently handles thresholds for
ordinal data, and means model for covariates matrices in the twin models if needed.

It's the replacement for \code{xmu_assemble_twin_supermodel} approach.
}
\details{
\code{xmu_make_TwinSuperModel} is used in twin models (e.g.\code{\link[=umxCP]{umxCP()}}, \code{\link[=umxACE]{umxACE()}} and \code{\link[=umxACEv]{umxACEv()}} and will be added to the other models: \code{\link[=umxGxE]{umxGxE()}}, \code{\link[=umxIP]{umxIP()}},
simplifying code maintenance.

It takes \code{mzData} and \code{dzData}, a list of the \code{selDVs} to analyse and optional \code{selCovs} (as well as \code{sep} and \code{nSib}), along with other
relevant information such as whether the user wants to \code{equateMeans}.
It can also handle a \code{weightVar}.

If covariates are passed in these are included in the means model (via a call to \code{xmuTwinUpgradeMeansToCovariateModel}.

\strong{Modeling}

\strong{Matrices created}

\emph{top model}

For raw and WLS data, \code{top} contains a \code{expMeans} matrix (if needed). For summary data, the top model contains only a name.

For ordinal data, \code{top} gains \code{top.threshMat} (from a call to \code{\link[=umxThresholdMatrix]{umxThresholdMatrix()}}).

For covariates, top stores the \code{intercepts} matrix and a \code{betaDef} matrix. These are then used to make expMeans in \code{MZ} and \code{DZ}.

\emph{MZ and DZ models}

\code{MZ} and \code{DZ} contain the data, and an expectation referencing \code{top.expCovMZ} and \code{top.expMean}, and, \code{vector = bVector}.
For continuous raw data, MZ and DZ contain \code{\link[OpenMx:mxExpectationNormal]{OpenMx::mxExpectationNormal()}} and \code{\link[OpenMx:mxFitFunctionML]{OpenMx::mxFitFunctionML()}}.
For WLS these the fit function is switched to \code{\link[OpenMx:mxFitFunctionWLS]{OpenMx::mxFitFunctionWLS()}} with appropriate \code{type} and \code{allContinuousMethod}.

For binary, a constraint and algebras are included to constrain \code{Vtot} (A+C+E) to 1.

If a \code{weightVar} is detected, these columns are used to create a row-weighted MZ and DZ models.

If \code{equateMeans} is \code{TRUE}, then the Twin-2 vars in the mean matrix are equated by label with Twin-1.

Decent starts are guessed from the data.
\code{varStarts} is computed as \code{sqrt(variance)/3} of the DVs and \code{meanStarts} as the variable means.
For raw data, a check is made for ordered variables. For Binary variables, means are fixed at 0 and
total variance (A+C+E) is fixed at 1. For ordinal variables, the first 2 thresholds are fixed.

Where needed, e.g. continuous raw data, top adds a means matrix "expMean".
For ordinal data, top adds a \code{\link[=umxThresholdMatrix]{umxThresholdMatrix()}}.

If binary variables are present, matrices and a constraint to hold \code{A+C+E == 1} are added to top.

If a weight variable is offered up, an \code{mzWeightMatrix} will be added.

\strong{Data handling}

In terms of data handling, \code{xmu_make_TwinSuperModel} was primarily designed to take
data.frames and process these into mxData.
It can also, however, handle cov and mxData input.

It can process data into all the types supported by \code{mxData}.

Raw data input with a target of \code{cov} or \code{cor} type requires the \code{numObsMZ} and \code{numObsDZ} to be set.

Type "WLS", "DWLS", or "ULS", data remain raw, but are handled as WLS in the \code{\link[OpenMx:mxFitFunctionWLS]{OpenMx::mxFitFunctionWLS()}}.

Unused columns are dropped.

If you pass in raw data, you can't request type cov/cor yet. Will work on this if desired.
}
\examples{
# ==============
# = Continuous =
# ==============
library(umx)
data(twinData)
twinData = umx_scale(twinData, varsToScale= c('ht1','ht2'))
mzData = twinData[twinData$zygosity \%in\%  "MZFF",] 
dzData = twinData[twinData$zygosity \%in\%  "DZFF",]
m1= xmu_make_TwinSuperModel(mzData=mzData, dzData=dzData, selDVs=c("wt","ht"), sep="", nSib=2)
names(m1) # "top" "MZ"  "DZ"
class(m1$MZ$fitfunction)[[1]] == "MxFitFunctionML"

# ====================
# = With a covariate =
# ====================

m1= xmu_make_TwinSuperModel(mzData=mzData, dzData=dzData, 
		selDVs= "wt", selCovs= "age", sep="", nSib=2)
m1$top$intercept$labels
m1$MZ$expMean

# ===============
# = WLS example =
# ===============
m1=xmu_make_TwinSuperModel(mzData=mzData, dzData=dzData,selDVs=c("wt","ht"),sep="",type="WLS")
class(m1$MZ$fitfunction)[[1]] == "MxFitFunctionWLS"
m1$MZ$fitfunction$type =="WLS"
# Check default all-continuous method
m1$MZ$fitfunction$continuousType == "cumulants"

# Choose non-default type (DWLS)
m1= xmu_make_TwinSuperModel(mzData= mzData, dzData= dzData,
	selDVs= c("wt","ht"), sep="", type="DWLS")
m1$MZ$fitfunction$type =="DWLS"
class(m1$MZ$fitfunction)[[1]] == "MxFitFunctionWLS"

# Switch WLS method
m1 = xmu_make_TwinSuperModel(mzData= mzData, dzData= dzData, selDVs= c("wt","ht"), sep= "",
  type = "WLS", allContinuousMethod = "marginals")
m1$MZ$fitfunction$continuousType == "marginals"
class(m1$MZ$fitfunction)[[1]] == "MxFitFunctionWLS"


# ============================================
# = Bivariate continuous and ordinal example =
# ============================================
data(twinData)
selDVs = c("wt", "obese")
# Cut BMI column to form ordinal obesity variables
ordDVs          = c("obese1", "obese2")
obesityLevels   = c('normal', 'overweight', 'obese')
cutPoints       = quantile(twinData[, "bmi1"], probs = c(.5, .2), na.rm = TRUE)
twinData$obese1 = cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 = cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
# Make the ordinal variables into mxFactors (ensure ordered is TRUE, and require levels)
twinData[, ordDVs] = umxFactor(twinData[, ordDVs])
mzData = twinData[twinData$zygosity \%in\%  "MZFF",] 
dzData = twinData[twinData$zygosity \%in\%  "DZFF",]
m1 = xmu_make_TwinSuperModel(mzData= mzData, dzData= dzData, selDVs= selDVs, sep="", nSib= 2)
names(m1) # "top" "MZ"  "DZ" 

# ==============
# = One binary =
# ==============
data(twinData)
cutPoints       = quantile(twinData[, "bmi1"], probs = .2, na.rm = TRUE)
obesityLevels   = c('normal', 'obese')
twinData$obese1 = cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 = cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
ordDVs = c("obese1", "obese2")
twinData[, ordDVs] = umxFactor(twinData[, ordDVs])
selDVs = c("wt", "obese")
mzData = twinData[twinData$zygosity \%in\% "MZFF",]
dzData = twinData[twinData$zygosity \%in\% "DZFF",]
m1 = xmu_make_TwinSuperModel(mzData= mzData, dzData= dzData, selDVs= selDVs, sep= "", nSib= 2)

# ========================================
# = Cov data (calls xmuTwinSuper_CovCor) =
# ========================================

data(twinData)
mzData =cov(twinData[twinData$zygosity \%in\% "MZFF", tvars(c("wt","ht"), sep="")], use="complete")
dzData =cov(twinData[twinData$zygosity \%in\% "DZFF", tvars(c("wt","ht"), sep="")], use="complete")
m1 = xmu_make_TwinSuperModel(mzData= mzData, dzData= dzData, selDVs= "wt", sep= "", 
	nSib= 2, numObsMZ = 100, numObsDZ = 100, verbose=TRUE)
class(m1$MZ$fitfunction)[[1]] =="MxFitFunctionML"
dimnames(m1$MZ$data$observed)[[1]]==c("wt1", "wt2")

}
\seealso{
Other xmu internal not for end user: 
\code{\link{umxModel}()},
\code{\link{umxRenameMatrix}()},
\code{\link{umxTwinMaker}()},
\code{\link{umx_APA_pval}()},
\code{\link{umx_fun_mean_sd}()},
\code{\link{umx_get_bracket_addresses}()},
\code{\link{umx_make}()},
\code{\link{umx_standardize}()},
\code{\link{umx_string_to_algebra}()},
\code{\link{umx}},
\code{\link{xmuHasSquareBrackets}()},
\code{\link{xmuLabel_MATRIX_Model}()},
\code{\link{xmuLabel_Matrix}()},
\code{\link{xmuLabel_RAM_Model}()},
\code{\link{xmuMI}()},
\code{\link{xmuMakeDeviationThresholdsMatrices}()},
\code{\link{xmuMakeOneHeadedPathsFromPathList}()},
\code{\link{xmuMakeTwoHeadedPathsFromPathList}()},
\code{\link{xmuMaxLevels}()},
\code{\link{xmuMinLevels}()},
\code{\link{xmuPropagateLabels}()},
\code{\link{xmuRAM2Ordinal}()},
\code{\link{xmuTwinSuper_Continuous}()},
\code{\link{xmuTwinUpgradeMeansToCovariateModel}()},
\code{\link{xmu_CI_merge}()},
\code{\link{xmu_CI_stash}()},
\code{\link{xmu_DF_to_mxData_TypeCov}()},
\code{\link{xmu_PadAndPruneForDefVars}()},
\code{\link{xmu_cell_is_on}()},
\code{\link{xmu_check_levels_identical}()},
\code{\link{xmu_check_needs_means}()},
\code{\link{xmu_check_variance}()},
\code{\link{xmu_clean_label}()},
\code{\link{xmu_data_missing}()},
\code{\link{xmu_data_swap_a_block}()},
\code{\link{xmu_describe_data_WLS}()},
\code{\link{xmu_dot_make_paths}()},
\code{\link{xmu_dot_make_residuals}()},
\code{\link{xmu_dot_maker}()},
\code{\link{xmu_dot_move_ranks}()},
\code{\link{xmu_dot_rank_str}()},
\code{\link{xmu_extract_column}()},
\code{\link{xmu_get_CI}()},
\code{\link{xmu_lavaan_process_group}()},
\code{\link{xmu_make_bin_cont_pair_data}()},
\code{\link{xmu_make_mxData}()},
\code{\link{xmu_match.arg}()},
\code{\link{xmu_name_from_lavaan_str}()},
\code{\link{xmu_path2twin}()},
\code{\link{xmu_path_regex}()},
\code{\link{xmu_safe_run_summary}()},
\code{\link{xmu_set_sep_from_suffix}()},
\code{\link{xmu_show_fit_or_comparison}()},
\code{\link{xmu_simplex_corner}()},
\code{\link{xmu_standardize_ACEcov}()},
\code{\link{xmu_standardize_ACEv}()},
\code{\link{xmu_standardize_ACE}()},
\code{\link{xmu_standardize_CP}()},
\code{\link{xmu_standardize_IP}()},
\code{\link{xmu_standardize_RAM}()},
\code{\link{xmu_standardize_SexLim}()},
\code{\link{xmu_standardize_Simplex}()},
\code{\link{xmu_start_value_list}()},
\code{\link{xmu_starts}()},
\code{\link{xmu_twin_add_WeightMatrices}()},
\code{\link{xmu_twin_check}()},
\code{\link{xmu_twin_get_var_names}()},
\code{\link{xmu_twin_upgrade_selDvs2SelVars}()}
}
\concept{xmu internal not for end user}
