% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BoundWaterEnvironments.R
\name{BoundWaterEnvironment.interact}
\alias{BoundWaterEnvironment.interact}
\title{Bound Water Environment (interactions)}
\usage{
BoundWaterEnvironment.interact(distances, set.oi.idc, names.atoms,
  names.res.atoms, radius = 3.6)
}
\arguments{
\item{distances}{Matrix of atomic pairwise distances}

\item{set.oi.idc}{Indices of protein atoms; can also HETATMs if those are of
interest}

\item{names.atoms}{Atom names from the PDB file in the PDB atomic naming
convention.}

\item{names.res.atoms}{Atom names of the form "RES AT"; created by combining
the residue and atom name while separating the two by a space. These do not
need to be unique because these names will be used to lookup hydrophilicity
values.}

\item{radius}{Distance in Angstroms between the atoms of interest; default:
3.6 Angstroms}
}
\value{
A list of the bound water environment values for nearby heavy atoms.
\itemize{
\item \strong{adn}: num of nearby heavy atoms
\item \strong{ahp.sum}: sum of hydrodrophilicy values
\item \strong{ahp.mu}: mean of hydrodrophilicy values
\item \strong{ahp.sd}: standard deviation of hydrodrophilicy values
\item \strong{hbonds}: number of possible hydrogen bonds
}
}
\description{
Various enviroment counts for bound waters.
}
\details{
For the heavy atoms near each water molecule (oxygen atom) the bound
water environment is calculated. These values are defined in the \strong{Return}
section. The default radius distance is 3.6 Angstroms. While it is possible
to define the radius to a value other than 3.6 this value is hardcoded into
the \code{\link[=ConservedWaters]{ConservedWaters()}} function. This might change in future versions.

\emph{\strong{NOTE}}: This function is designed to work with \code{\link[=ConservedWaters]{ConservedWaters()}} via
the \code{\link[base:apply]{base::apply()}} function processing rows (the \code{MARGIN = 1} option). For
this reason it is \strong{NOT} a public function. The \code{\link[=Nearby]{Nearby()}} is specifically
designed to work with this function.
}
\examples{
  \dontrun{
  distances <- PDB.1hai.h2o.prot.dists[3, ]
  set.oi.idc <- prot.idc
  names.atoms <- PDB.1hai.aoi.clean$elety[prot.idc]
  names.res.atoms <- paste(PDB.1hai.aoi.clean$resid[prot.idc], names.atoms, sep =" ")
  BoundWaterEnvironment.interact(distances,
                                 set.oi.idc,
                                 names.atoms,
                                 names.res.atoms,
                                 radius = 3.6)
  # $adn
  # [1] 9
  #
  # $ahp.sum
  # [1] 2.001
  #
  # $ahp.mu
  # [1] 0.2223
  #
  # $ahp.sd
  # [1] 0.2229
  #
  # $hbonds
  # [1] 4
  }

}
\references{
Paul C Sanschagrin and Leslie A Kuhn. Cluster analysis of
consensus water sites in thrombin and trypsin shows conservation between
serine proteases and contributions to ligand specificity. \emph{Protein
Science}, 1998, \strong{7} (\emph{10}), pp 2054-2064.
\href{http://doi.org/10.1002/pro.5560071002}{DOI: 10.1002/pro.5560071002}
\href{http://www.ncbi.nlm.nih.gov/pubmed/9792092}{PMID: 9792092}
\href{http://www.kuhnlab.bmb.msu.edu/software/watch/index.html}{WatCH webpage}

Leslie A Kuhn, Craig A Swanson, Michael E Pique, John A Tainer,
and Elizabeth D Getzof. Atomic and Residue Hydrophilicity in the Context of
Folded Protein Structures. \emph{PROTEINS: Structure, Function, and
Genetics}, 1995, \strong{2} (\emph{4}), pp 536-547.
\href{http://doi.org/10.1002/prot.340230408}{DOI: 10.1002/prot.340230408}
\href{http://www.ncbi.nlm.nih.gov/pubmed/8749849}{PMID: 8749849}
}
\seealso{
Other "Bound Water Environment": \code{\link{BoundWaterEnvironment.quality}},
  \code{\link{BoundWaterEnvironment}},
  \code{\link{Mobility}}, \code{\link{NormalizedBvalue}},
  \code{\link{calcBvalue}},
  \code{\link{calcNearbyHydrationFraction}},
  \code{\link{calcNumHydrogenBonds}}
}
\author{
Emilio Xavier Esposito \email{emilio@exeResearch.com}
}
