#' Local Variable Importance measure based on Ceteris Paribus profiles.
#'
#' This function calculate local importance measure in eight variants. We obtain eight variants measure through the possible options of three parameters such as `absolute_deviation`, `point` and `density`.
#'
#' @param profiles data.frame generated by \code{ingredients::ceteris_paribus()}
#' @param data data.frame with raw data to model
#' @param absolute_deviation logical parameter, if `absolute_deviation = TRUE` then measure is calculated as absolute deviation, else is calculated as a root from average squares
#' @param point logical parameter, if `point = TRUE` then measure is calculated as a distance from f(x), else measure is calculated as a distance from average profiles
#' @param density logical parameter, if `density = TRUE` then measure is weighted based on the density of variable, else is not weighted
#'
#' @return A data.frame of the class 'local_variable_importance'.
#' It's a data.frame with calculated local variable importance measure.
#' @examples
#'
#'
#' library("DALEX")
#' data(apartments)
#'
#' library("randomForest")
#' apartments_rf_model <- randomForest(m2.price ~ construction.year + surface +
#'                                     floor + no.rooms, data = apartments)
#'
#' explainer_rf <- explain(apartments_rf_model, data = apartmentsTest[,2:5],
#'                         y = apartmentsTest$m2.price)
#'
#' new_apartment <- data.frame(construction.year = 1998, surface = 88, floor = 2L, no.rooms = 3)
#'
#' library("ingredients")
#' profiles <- ceteris_paribus(explainer_rf, new_apartment)
#'
#' library("vivo")
#' local_variable_importance(profiles, apartments[,2:5],
#'                           absolute_deviation = TRUE, point = TRUE, density = TRUE)
#'
#' local_variable_importance(profiles, apartments[,2:5],
#'                           absolute_deviation = TRUE, point = TRUE, density = FALSE)
#'
#' local_variable_importance(profiles, apartments[,2:5],
#'                           absolute_deviation = TRUE, point = FALSE, density = TRUE)
#'
#'
#'
#' @export
#'


local_variable_importance <- function(profiles, data, absolute_deviation = TRUE, point = TRUE, density = TRUE){
  if (!(c("ceteris_paribus_explainer") %in% class(profiles)))
    stop("The local_variable_importance() function requires an object created with ceteris_paribus() function.")
  if (!c("data.frame") %in% class(data))
    stop("The local_variable_importance() function requires a data.frame.")

  avg_yhat <- lapply(unique(profiles$`_vname_`), function(x){
    mean(profiles$`_yhat_`[profiles$`_vname_` == x])
  })
  names(avg_yhat) <- unique(profiles$`_vname_`)

  variable_split <- vivo::calculate_variable_split(data, variables = colnames(data))

  weight <- vivo::calculate_weight(profiles, data, variable_split = variable_split)

  obs <- attr(profiles, "observations")


  if(absolute_deviation == TRUE){
    if(point == TRUE){
      if(density == TRUE){
        result <- unlist(lapply(unique(profiles$`_vname_`), function(m){
          sum(abs(weight[[m]] * (profiles[profiles$`_vname_` == m, "_yhat_"] - unlist(unname(obs["_yhat_"])))))
        }))
      }else{
        result <- unlist(lapply(unique(profiles$`_vname_`), function(w){
          sum(abs((profiles[profiles$`_vname_` == w, "_yhat_"] - unlist(unname(obs["_yhat_"])))))
        }))
      }
    }else{
      if(density == TRUE){
        result <- unlist(lapply(unique(profiles$`_vname_`), function(m){
          sum(abs(weight[[m]] *(profiles[profiles$`_vname_` == m, "_yhat_"] - avg_yhat[[m]])))
        }))
      }else{
        result <- unlist(lapply(unique(profiles$`_vname_`), function(w){
          sum(abs((profiles[profiles$`_vname_` == w, "_yhat_"] - avg_yhat[[w]])))
        }))
      }
    }
  }else{
    if(point == TRUE){
      if(density == TRUE){
        result <- unlist(lapply(unique(profiles$`_vname_`), function(m){
          sqrt(sum((weight[[m]] * (profiles[profiles$`_vname_` == m, "_yhat_"] - unlist(unname(obs["_yhat_"]))))^2)/length(profiles[profiles$`_vname_` == m, "_yhat_"]))
        }))
      }else{
        result <- unlist(lapply(unique(profiles$`_vname_`), function(w){
          sqrt(sum((profiles[profiles$`_vname_` == w, "_yhat_"] - unlist(unname(obs["_yhat_"])))^2)/length(profiles[profiles$`_vname_` == w, "_yhat_"]))
        }))
      }
    }else{
      if(density == TRUE){
        result <- unlist(lapply(unique(profiles$`_vname_`), function(m){
          sqrt(sum((weight[[m]] * (profiles[profiles$`_vname_` == m, "_yhat_"] - avg_yhat[[m]]))^2)/length(profiles[profiles$`_vname_` == m, "_yhat_"]))
        }))
      }else{
        result <- unlist(lapply(unique(profiles$`_vname_`), function(w){
          sqrt(sum((profiles[profiles$`_vname_` == w, "_yhat_"] - avg_yhat[[w]])^2)/(length(profiles[profiles$`_vname_` == w, "_yhat_"])))
        }))
      }
    }
  }

  lvivo = data.frame(variable_name = unique(profiles$`_vname_`), measure = result)
  class(lvivo) = c("local_importance", "data.frame")
  lvivo
}

