#' Read tabulated files imported from Thermo ProteomeDiscoverer
#'
#' Quantification results form \href{https://www.thermofisher.com/order/catalog/product/OPTON-30812}{Thermo ProteomeDiscoverer} 
#' exported as tabulated text can be imported and relevant information extracted. 
#' The final output is a list containing 3 elements: \code{$annot}, \code{$raw} and optional \code{$quant}, or returns data.frame with entire content of file if \code{separateAnnot=FALSE}.
#' This function has been developed using MaxQuant version Thermo ProteomeDiscoverer2.4, the format of resulting file is typically well conserved.  
#' 
#' @param fileName (character) name of file to be read (default 'proteinGroups.txt' as typically generated by MaxQuant in txt folder) 
#' @param path (character) path of file to be read
#' @param normalizeMeth (character) normalization method (will be sent to  \code{\link[wrMisc]{normalizeThis}}) 
#' @param annotCol (character) column names to be read 
#' @param quantCol (character or integer) exact col-names, or if length=1 content of \code{quantCol} will be used as pattern to search among column-names for $quant using \code{grep} 
#' @param refLi (integer) custom decide which line of data is main species
#' @param separateAnnot (logical) if \code{TRUE} output will be organized as list with \code{$annot}, \code{$abund} for initial/raw abundance values and \code{$quant} with final normalized quantitations 
#' @param graphTit (character) custom title to plot
#' @param specPref (character) define characteristic text for recognizing the 3 (main) groups of species (contaminants,mainSpecies,spike)
#' @param separateAnnot (logical) if \code{TRUE} output will be organized as list with \code{$annot}, \code{$abund} for initial/raw abundance values and \code{$quant} with final normalized quantitations
#' @param plotGraph (logical) optional plot of type vioplot of initial and normalized data (using \code{normalizeMeth}); if integer, it will be passed to \code{layout} when plotting
#' @param silent (logical) suppress messages
#' @param callFrom (character) allow easier tracking of message(s) produced
#' @return list with \code{$annot}, \code{$raw} for initial/raw abundance values and \code{$quant} with final normalized quantitations, or returns data.frame with annot and quant if \code{separateAnnot=FALSE}
#' @seealso \code{\link[utils]{read.table}}, \code{\link[wrMisc]{normalizeThis}}) , \code{\link{readProlineFile}} 
#' @examples
#' path1 <- system.file("extdata",package="wrProteo")
#' fiNa <- "exampleProtDiscov1.txt"
#' dataPD <- readPDExport(file=fiNa,path=path1)
#' summary(dataPD$quant)
#' matrixNAinspect(dataPD$quant,gr=gl(2,3)) 
#' 
#' @export
readPDExport <- function(fileName,path=NULL,normalizeMeth="median",
  annotCol=c("Accession","Description","Gene","Sum.PEP.Score","Coverage....","X..Peptides","X..PSMs","X..Unique.Peptides", "X..AAs","MW..kDa."),
  quantCol="^S", refLi=NULL, separateAnnot=TRUE,plotGraph=TRUE, graphTit ="Proteome Discoverer",
  specPref=c(conta="CON_|LYSC_CHICK",mainSpecies="OS=Saccharomyces cerevisiae",spike="HUMAN_UPS"), silent=FALSE,callFrom=NULL) {
  ## read ProteomeDiscoverer exported txt
  fxNa <- wrMisc::.composeCallName(callFrom,newNa="readPDExport")
  opar <- graphics::par(no.readonly=TRUE)
  ## check & read file
  chPa <- length(grep("/",fileName)) >0 | length(grep("\\\\",fileName)) >0       # check for path already in fileName "
  if(length(path) <1) path <- "."
  paFi <- if(!chPa) file.path(path[1],fileName[1]) else fileName[1]              # use path only when no path joined to fileName
  chFi <- file.exists(paFi)
  if(!chFi) stop(" file ",fileName," was NOT found ",if(chPa) paste(" in path ",path)," !")
  if(length(grep("\\.txt$",fileName)) <1) message(fxNa," Trouble ahead, expecting tabulated text file (this file might not be right format) !!") 
  tmp <- utils::read.delim(file.path(paFi),stringsAsFactors=FALSE)
  ## locate & extract annotation 
  if(length(annotCol) <1) annotCol <- c("Accession","Description","Gene","Sum.PEP.Score","Coverage....","X..Peptides","X..PSMs","X..Unique.Peptides", "X..AAs","MW..kDa.")
  annotColNo <- match(annotCol,colnames(tmp))
  ## check for R-friendly export
  specRepl <- cbind(ini=c("Coverage...."),new=c("Coverage.in.Percent"))
  annotCol2 <- unique(c(sub("X\\.\\.","Number.of.",annotCol), apply(specRepl,1,function(x) sub(x[1],x[2],annotCol)) ))
  annotColN2 <- match(annotCol2,colnames(tmp))
  if(sum(!is.na(annotColN2)) > sum(!is.na(annotColNo))) { annotColNo <- annotColN2
    annotCol <- annotCol2
    if(!silent) message(fxNa," correcting 'annotCol' to export of R-frendly colnames")}  
  if(all(is.na(annotColNo))) stop(" Problem with 'annotCol' : Could NOT find any annotation-column")
  if(any(is.na(annotColNo))) {if(annotCol != annotCol2) message(fxNa,"Can't find columns ",wrMisc::pasteC(annotCol[is.na(annotColNo)],quote="'"))
    annotCol <- annotCol[!is.na(annotColNo)] }
  annot <- as.matrix(tmp[,wrMisc::naOmit(annotColNo)])
  chUniq <- apply(annot,2,function(x) sum(duplicated(x)))
  tmp <- tmp[,-wrMisc::naOmit(annotColNo)]
  IDs <- if(any(chUniq <1)) annot[,which.max(chUniq <1)] else wrMisc::correctToUnique(annot[,which.min(chUniq)])
  if(all(chUniq >0)) {
    if(!silent) message(fxNa,"could not find any unique column content for direct use as ID !!")}
  rownames(tmp) <- rownames(annot) <- IDs  
  specPD0 <- list(conta=grep(specPref[1],annot[,annotCol[2]]),
    mainSpe=grep(specPref[2],annot[,annotCol[2]]),
    spike= grep(specPref[3],annot[,annotCol[2]] ))
  specPD <- rep(NA,nrow(tmp))
  for(i in 1:length(specPD0)) if(length(specPD0[[i]]) >0) specPD[specPD0[[i]]] <- names(specPD0)[i]
  annot <- cbind(Spec=specPD,annot)                                             # may be better to name column 'species' ?
  ## locate & extract abundance/quantitation data
  if(length(quantCol) >1) { abund <- as.matrix(wrMisc::extrColsDeX(tmp,extrCol=quantCol,doExtractCols=TRUE,callFrom=fxNa))
  } else {
    quantCol <-  grep(quantCol,colnames(tmp)) 
    chNa <- is.na(quantCol)
    if(all(chNa)) stop("Could not find any of of the columns specified in argument 'quantCol' !")
    if(any(chNa)) { 
      if(!silent) message(fxNa," Could not find columns ",wrMisc::pasteC(quantCol[which(chNa)],quote="'")," .. omit")
      quantCol <- wrMisc::naOmit(quantCol)} 
    abund <- as.matrix(tmp[,quantCol]) }                                        # abundance val
  chNum <- is.numeric(abund)
  if(!chNum) {abund <- apply(tmp[,quantCol],2,wrMisc::convToNum,convert="allChar",callFrom=fxNa)}
  if(is.character(refLi) & length(refLi)==1) refLi <- which(annot[,"Spec"]==refLi)   # may be "mainSpe"
  quant <- wrMisc::normalizeThis(log2(abund),meth=normalizeMeth,refLi=refLi,callFrom=fxNa) 
  ## plot distribution of intensities
  if(length(plotGraph) >0) {if(is.numeric(plotGraph)) {custLay <- plotGraph; plotGraph <- TRUE
    } else  {plotGraph <- as.logical(plotGraph[1])}}
  if(plotGraph){
    graphics::par(mar=c(3, 3, 3, 1))                          # mar: bot,le,top,ri
    graphics::layout(1:2)
    graphics::boxplot(log2(abund),main=paste(graphTit,"(initial)",sep=" "),las=1,outline=FALSE) 
    graphics::abline(h=round(log2(stats::median(abund,na.rm=TRUE)))+c(-1,0,1),lty=2,col=grDevices::grey(0.6)) 
    ## plot normalized
    graphics::boxplot(quant,main=paste(graphTit,"(normalized)",sep=" "),las=1,outline=FALSE) 
    graphics::abline(h=round(stats::median(quant,na.rm=TRUE))+c(-1,0,1),lty=2,col=grDevices::grey(0.6)) 
    on.exit(graphics::par(opar)) 
    }
  if(separateAnnot) list(raw=abund,quant=quant,annot=annot) else data.frame(quant,annot)  
}
    
