#' Derive the importance of a predictor used in the PNN 
#'
#' The function \code{pnn.x_imp} derives the importance of a predictor used in the PNN,
#' where the "importance" is measured by the increase in cross entropy after eliminating the impact 
#' of the predictor in interest.
#'
#' @param net A PNN object generated by pnn.fit() 
#' @param i   The ith predictor in the PNN
#'
#' @return A vector with the variable name and two values of importance measurements, namely "imp1" and "imp2".
#'         The "imp1" measures the increase in cross entropy after replacing all values of the predictor with its mean.
#'         The "imp2" measures the increase in cross entropy after dropping the predictor from the PNN.
#'
#' @seealso \code{\link{pnn.x_pfi}}
#'
#' @examples
#' data(iris, package = "datasets")
#' Y <- iris[, 5]
#' X <- scale(iris[, 1:4])
#' pnet <- pnn.fit(x = X, y = Y)
#' pnn.x_imp(pnet, 1)

pnn.x_imp <- function(net, i) {
  if (class(net) != "Probabilistic Neural Net") stop("net needs to be a PNN.", call. = F)
  if (i > ncol(net$x)) stop("the selected variable is out of bound.", call. = F)

  xname <- colnames(net$x)[i]
  x <- net$x
  x[, i] <-  rep(mean(net$x[, i]), nrow(net$x))
  ll0 <- logl(y_pred = pnn.predict(net, net$x), y_true = net$y.ind)
  ll1 <- logl(y_pred = pnn.predict(net, x), y_true = net$y.ind)
  ll2 <- logl(y_pred = pnn.predict(pnn.fit(x = x[, -i], y = net$y.raw, sigma = net$sigma), x[, -i]), 
              y_true = net$y.ind)
  imp1 <- round(max(0, ll1 / ll0 - 1), 8)
  imp2 <- round(max(0, ll2 / ll0 - 1), 8)
  return(data.frame(var = xname, imp1 = imp1, imp2 = imp2))
}
