% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bdEigen_hdf5}
\alias{bdEigen_hdf5}
\title{Eigenvalue Decomposition for HDF5-Stored Matrices using Spectra}
\usage{
bdEigen_hdf5(
  filename,
  group = NULL,
  dataset = NULL,
  k = NULL,
  which = NULL,
  ncv = NULL,
  bcenter = NULL,
  bscale = NULL,
  tolerance = NULL,
  max_iter = NULL,
  compute_vectors = NULL,
  overwrite = NULL,
  threads = NULL
)
}
\arguments{
\item{filename}{Character string. Path to the HDF5 file containing the input matrix.}

\item{group}{Character string. Path to the group containing the input dataset.}

\item{dataset}{Character string. Name of the input dataset to decompose.}

\item{k}{Integer. Number of eigenvalues to compute (default = 6, following Spectra convention).}

\item{which}{Character string. Which eigenvalues to compute (default = "LM"):
\itemize{
\item "LM": Largest magnitude
\item "SM": Smallest magnitude
\item "LR": Largest real part (non-symmetric matrices)
\item "SR": Smallest real part (non-symmetric matrices)
\item "LI": Largest imaginary part (non-symmetric matrices)
\item "SI": Smallest imaginary part (non-symmetric matrices)
\item "LA": Largest algebraic (symmetric matrices)
\item "SA": Smallest algebraic (symmetric matrices)
}}

\item{ncv}{Integer. Number of Arnoldi vectors (default = 0, auto-selected as max(2*k+1, 20)).}

\item{bcenter}{Logical. If TRUE, centers the data by subtracting column means (default = FALSE).}

\item{bscale}{Logical. If TRUE, scales the centered columns by their standard deviations (default = FALSE).}

\item{tolerance}{Numeric. Convergence tolerance for Spectra algorithms (default = 1e-10).}

\item{max_iter}{Integer. Maximum number of iterations for Spectra algorithms (default = 1000).}

\item{compute_vectors}{Logical. If TRUE (default), computes both eigenvalues and eigenvectors.}

\item{overwrite}{Logical. If TRUE, allows overwriting existing results (default = FALSE).}

\item{threads}{Integer. Number of threads for parallel computation (default = NULL, uses available cores).}
}
\value{
List with components:
\describe{
\item{fn}{Character string with the HDF5 filename}
\item{values}{Character string with the full dataset path to the eigenvalues (real part) (group/dataset)}
\item{vectors}{Character string with the full dataset path to the eigenvectors (real part) (group/dataset)}
\item{values_imag}{Character string with the full dataset path to the eigenvalues (imaginary part), or NULL if all eigenvalues are real}
\item{vectors_imag}{Character string with the full dataset path to the eigenvectors (imaginary part), or NULL if all eigenvectors are real}
\item{is_symmetric}{Logical indicating whether the matrix was detected as symmetric}
}
}
\description{
Computes the eigenvalue decomposition of a large matrix stored in an HDF5 file using
the Spectra library. This provides consistent results with the RSpectra package and
can handle both symmetric and non-symmetric matrices.
}
\details{
This function uses the Spectra library (same as RSpectra) for eigenvalue computation,
ensuring consistent results. Key features include:
\itemize{
\item Automatic detection of symmetric vs non-symmetric matrices
\item Support for both real and complex eigenvalues/eigenvectors
\item Memory-efficient block-based processing for large matrices
\item Parallel processing support
\item Various eigenvalue selection criteria
\item Consistent interface with RSpectra::eigs()
}

The implementation automatically:
\itemize{
\item Detects matrix symmetry and uses appropriate solver (SymEigsSolver vs GenEigsSolver)
\item Handles complex eigenvalues for non-symmetric matrices
\item Saves imaginary parts separately when non-zero
\item Provides the same results as RSpectra::eigs() function
}
}
\examples{
\dontrun{
library(BigDataStatMeth)
library(rhdf5)
library(RSpectra)

# Create a sample matrix (can be non-symmetric)
set.seed(123)
A <- matrix(rnorm(2500), 50, 50)

fn <- "test_eigen.hdf5"
bdCreate_hdf5_matrix_file(filename = fn, object = A, group = "data", dataset = "matrix")

# Compute eigendecomposition with BigDataStatMeth
res <- bdEigen_hdf5(fn, "data", "matrix", k = 6, which = "LM")

# Compare with RSpectra (should give same results)
rspectra_result <- eigs(A, k = 6, which = "LM")

# Extract results from HDF5
eigenvals_bd <- h5read(res$fn, res$values)
eigenvecs_bd <- h5read(res$fn, res$vectors)

# Compare eigenvalues (should be identical)
all.equal(eigenvals_bd, Re(rspectra_result$values), tolerance = 1e-12)

# For non-symmetric matrices, check imaginary parts
if (!is.null(res$values_imag)) {
  eigenvals_imag <- h5read(res$fn, res$values_imag)
  all.equal(eigenvals_imag, Im(rspectra_result$values), tolerance = 1e-12)
}

# Remove file
if (file.exists(fn)) {
  file.remove(fn)
}
}

}
\references{
\itemize{
\item Qiu, Y., & Mei, J. (2022). RSpectra: Solvers for Large-Scale Eigenvalue and SVD Problems.
\item Li, R. (2021). Spectra: C++ Library For Large Scale Eigenvalue Problems.
}
}
\seealso{
\itemize{
\item \code{\link{bdSVD_hdf5}} for Singular Value Decomposition
\item \code{\link{bdPCA_hdf5}} for Principal Component Analysis
\item \code{RSpectra::eigs} for the R equivalent function
}
}
