## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(E2E)

## ----include=FALSE------------------------------------------------------------
# Set up a 2-core cluster for parallel processing in this vignette
# This is crucial for passing R CMD check on CI/CD platforms
cl <- parallel::makeCluster(2)
doParallel::registerDoParallel(cl)

## -----------------------------------------------------------------------------
initialize_modeling_system_dia()

## -----------------------------------------------------------------------------
# To run all, use model = "all_dia".
results_all_dia <- models_dia(train_dia, model = c("rf", "lasso", "xb"))

# Print a summary for a specific model (e.g., Random Forest)
print_model_summary_dia("rf", results_all_dia$rf)

## -----------------------------------------------------------------------------
# Run a specific subset of models with tuning enabled and custom thresholds
results_dia_custom <- models_dia(
  data = train_dia,
  model = c("rf", "lasso", "xb"),
  tune = TRUE,
  seed = 123,
  threshold_choices = list(rf = "f1", lasso = 0.6, xb = "youden")
)

# View the custom results
print_model_summary_dia("rf", results_dia_custom$rf)

## -----------------------------------------------------------------------------
# Create a Bagging ensemble with RandomForest as the base model
# n_estimators is reduced for faster execution in this example.
bagging_rf_results <- bagging_dia(train_dia, base_model_name = "rf", tune_base_model = FALSE, n_estimators = 5)
print_model_summary_dia("Bagging (RandomForest)", bagging_rf_results)

## -----------------------------------------------------------------------------
# Create a soft voting ensemble from the top models
voting_soft_results <- voting_dia(
  results_all_models = results_all_dia,
  data = train_dia,
  type = "soft"
)
print_model_summary_dia("Voting (Soft)", voting_soft_results)

## -----------------------------------------------------------------------------
# Create a Stacking ensemble with Lasso as the meta-model
stacking_lasso_results <- stacking_dia(
  results_all_models = results_all_dia,
  data = train_dia,
  meta_model_name = "lasso"
)
print_model_summary_dia("Stacking (Lasso)", stacking_lasso_results)

## -----------------------------------------------------------------------------
# Create an EasyEnsemble with RandomForest as the base model
# n_estimators is reduced for faster execution.
results_imbalance_dia <- imbalance_dia(train_dia, base_model_name = "rf", n_estimators = 5)
print_model_summary_dia("Imbalance (Random Forest)", results_imbalance_dia)

## -----------------------------------------------------------------------------
# Apply the trained Bagging model to the test set
bagging_pred_new <- apply_dia(
  trained_model_object = bagging_rf_results$model_object,
  new_data = test_dia,
  label_col_name = "outcome"
)

# Evaluate these new predictions
eval_results_new <- evaluate_predictions_dia(
  prediction_df = bagging_pred_new,
  threshold_choices = "f1")
print(eval_results_new)

## ----fig.width=5, fig.height=5, warning=FALSE---------------------------------
# ROC Curve
p1 <- figure_dia(type = "roc", data = results_imbalance_dia)
#plot(p1)

# Precision-Recall Curve
p2 <- figure_dia(type = "prc", data = results_imbalance_dia)
#plot(p2)

# Confusion Matrix
p3 <- figure_dia(type = "matrix", data = results_imbalance_dia)
#plot(p3)

## ----include=FALSE------------------------------------------------------------
# Stop the parallel cluster
parallel::stopCluster(cl)

