
sample_limit_barycenter <- \(gen_G, dual_constr, solver) {
    had_deriv_ot_barycenter(gen_G(), dual_constr, solver)
}

sample_finite_barycenter <- \(mu, w, costm, n, constr_mat, solver) {
    nu <- emp_measure(n, mu)
    ot_barycenter(nu, costm, w, solver = solver, constr_mat = constr_mat)$cost
}

##

#' @title Simulations for `ot_barycenter_test`
#' @description Perform simulations for the test statistic used in [`ot_barycenter_test`].
#' @param mu matrix (row-wise) or list containing \eqn{K} probability vectors.
#' @param costm semi-metric cost matrix \eqn{c \in \mathbb{R}^{N \times N}}.
#' @param n vector of samples sizes.
#' @param w weight vector \eqn{w \in \mathbb{R}^K_+}.
#' @param delta vector of asymptotic sample size ratios.
#' @param num.sim number of samples to draw from the limiting null distribution.
#' @param solver the LP solver to use, see [`ot_test_lp_solver`].
#' @param mean mean of the Gaussians appearing in the limiting distribution. Must be of the same structure as `mu`.
#' @details See [`ot_barycenter_test`] for the definition of the test statistic and more details.
#'
#' `simulate_finite_ot_barycenter_test` simulates from the finite sample distribution.
#'
#' `simulate_limit_ot_barycenter_test_null` and `simulate_limit_ot_barycenter_test_alt` simulate from the limiting distribution
#' under the null or alternative, respectively.
#'
#' All these simulations can be done in parallel via [`future::plan`] and the progress can be shown with [`progressr::with_progress`].
#' @returns A vector containing the simulated samples.
#' @rdname sim_barycenter
#' @order 2
#' @example examples/sim_bary.R
#' @export
simulate_limit_ot_barycenter_test_null <- \(mu, costm, n = NULL, delta = NULL, w = NULL, num.sim = 1000, solver = ot_test_lp_solver(), mean = NULL) {
    check_numsim(num.sim)
    check_lp_solver(solver)
    mu <- get_fac_mat(mu, TRUE)
    check_mu(mu)
    K <- nrow(mu)
    N <- ncol(mu)
    check_cost_mat(costm, N)
    delta <- unlist(delta)
    check_delta(delta, K)
    if (is.null(w)) {
        w <- rep(1 / K, K)
    } else {
        check_w(w, K)
    }

    if (!is.null(n)) {
        warning("The parameter n is currently unused.")
    }
    if (!is.null(mean)) {
        mean <- get_fac_mat(mean)
        stopifnot(is_num_mat(mean), dim(mean) == dim(mu))
    }

    # make sure null holds
    mu <- matrix(colMeans(mu), K, N, byrow = TRUE)
    gen_G <- add_mean_to_gen(get_gen_G_mat_one, mean)(mu, delta, N, K)
    dual_constr <- had_deriv_ot_barycenter_null_constr(mu, w, costm, N, K)

    p <- progressr::progressor(steps = num.sim)
    future_replicate(
        num.sim,
        { x <- sample_limit_barycenter(gen_G, dual_constr, solver); p(); x }
    )
}

#' @rdname sim_barycenter
#' @order 3
#' @export
simulate_limit_ot_barycenter_test_alt <- \(mu, costm, delta, w = NULL, num.sim = 1000, solver = ot_test_lp_solver()) {
    check_numsim(num.sim)
    check_lp_solver(solver)
    mu <- get_fac_mat(mu, TRUE)
    check_mu(mu)
    K <- nrow(mu)
    N <- ncol(mu)
    check_cost_mat(costm, N)
    delta <- unlist(delta)
    check_delta(delta, K)
    if (is.null(w)) {
        w <- rep(1 / K, K)
    } else {
        check_w(w, K)
    }

    objval <- ot_barycenter(mu, costm, w, solver = solver)$cost

    gen_G <- get_gen_G(mu, delta, N, K)
    dual_constr <- had_deriv_ot_barycenter_alt_constr(mu, w, costm, objval)

    p <- progressr::progressor(steps = num.sim)
    future_replicate(
        num.sim,
        { x <- sample_limit_barycenter(gen_G, dual_constr, solver); p(); x }
    )
}

#' @rdname sim_barycenter
#' @order 1
#' @export
simulate_finite_ot_barycenter_test <- \(mu, costm, n, w = NULL, num.sim = 1000, solver = ot_test_lp_solver()) {
    check_numsim(num.sim)
    check_lp_solver(solver)
    mu <- get_fac_mat(mu)
    check_mu(mu)
    K <- nrow(mu)
    N <- ncol(mu)
    check_cost_mat(costm, N)
    n <- unlist(n)
    check_n(n, K)
    if (is.null(w)) {
        w <- rep(1 / K, K)
    } else {
        check_w(w, K)
    }

    rho <- limit_coeffs(n)$rho
    constr_mat <- ot_barycenter_constrmat(K, N)
    objval <- ot_barycenter(mu, costm, w, solver = solver, constr_mat = constr_mat)$cost

    p <- progressr::progressor(steps = num.sim)
    ls <- future_replicate(
        num.sim,
        { x <- sample_finite_barycenter(mu, w, costm, n, constr_mat, solver); p(); x }
    )
    sqrt(rho) * (ls - objval)
}
