# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_gdp

library(testthat)

test_that("get_japan_gdp() returns a tibble with correct structure", {
  skip_on_cran()

  result <- get_japan_gdp()

  # Should return a tibble or NULL (in case API fails)
  expect_true(is.null(result) || inherits(result, "tbl_df"))

  # If NULL returned, stop further checks
  if (is.null(result)) {
    succeed("API returned NULL, structure test skipped.")
    return()
  }

  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 5)

  expect_named(result,
               c("indicator", "country", "year", "value", "value_label"))

  # Types
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_true(is.numeric(result$value))
  expect_type(result$value_label, "character")
})

test_that("get_japan_gdp() returns correct years and row count", {
  skip_on_cran()

  result <- get_japan_gdp()
  if (is.null(result)) {
    succeed("API returned NULL, year test skipped.")
    return()
  }

  expect_equal(nrow(result), 13)
  expect_equal(sort(result$year), 2010:2022)

  # Sorted descending (2022 → 2010)
  expect_true(all(diff(result$year) <= 0))
})

test_that("get_japan_gdp() returns valid country and indicator information", {
  skip_on_cran()

  result <- get_japan_gdp()
  if (is.null(result)) {
    succeed("API returned NULL, content test skipped.")
    return()
  }

  expect_true(all(result$country == "Japan"))
  expect_true(all(grepl("GDP", result$indicator)))
})

test_that("get_japan_gdp() returns numeric GDP values in expected ranges", {
  skip_on_cran()

  result <- get_japan_gdp()
  if (is.null(result)) {
    succeed("API returned NULL, numeric test skipped.")
    return()
  }

  # All values must be positive
  expect_true(all(result$value > 0, na.rm = TRUE))

  # Japan GDP is typically between 3e12 and 7e12
  expect_true(all(result$value < 1e13))
})

test_that("get_japan_gdp() has correctly formatted value labels", {
  skip_on_cran()

  result <- get_japan_gdp()
  if (is.null(result)) {
    succeed("API returned NULL, label test skipped.")
    return()
  }

  # All labels should contain commas
  expect_true(all(grepl(",", result$value_label)))
})

test_that("get_japan_gdp() handles API errors gracefully", {
  skip_on_cran()

  result <- tryCatch(get_japan_gdp(), error = function(e) NULL)

  expect_true(is.null(result) || inherits(result, "tbl_df"))
})
