#===============================================================================
#                      Mann-Kendall Test Functions
#===============================================================================

#' @title Univariate Mann-Kendall Test
#'
#' @description The functions performs the univariate Mann-Kendall test.
#'
#' @param x Numeric vector representing a data series.
#'
#' @return A named list
#'
#' - `statistic`, the estimated Mann-Kendall test statistics.
#' - `p.value`, the estimated p-value for the test.
#'
#' @details
#' The univariate Mann-Kendall (MK) test is used to detect monotonic trends in a univariate data series.
#' It tests the null hypothesis (\eqn{h_0}) of no trend, against an alternative.
#'
#' Let \eqn{(x_1, x_2, ..., x_n)} be a data series of length \eqn{n}, the MK test statistic is
#' given by:
#' \deqn{M = \sum_{i=1}^{n-1} \sum_{j=i+1}^{n} sgn(x_j-x_i)}
#' where \eqn{sgn(\cdot)} is the sign function:
#' \deqn{
#' sgn(x)=\begin{cases}
#' -1 \quad \text{if } x<0, \\
#' 0 \quad \text{if } x=0, \\
#' +1 \quad \text{if } x>0
#' \end{cases}
#' }
#' Under \eqn{H_0} the test statistic is asymptotically normally distributed with mean and
#' variance:
#' \deqn{E(M) = 0}
#' \deqn{\text{Var}(M)=\frac{n(n-1)(n+5)}{18}}
#'
#'
#' @references
#'
#' - Hamed, K.H., Rao, A.R., 1998. A modified Mann-Kendall trend test for autocorrelated
#'    data. J. Hydrol. 204 (1–4), 182–196.
#' - Kendall, M., (1975). Rank Correlation Methods; Griffin: London.
#'
#' @examples
#' # Sample data (Both marginal distributions have trend):
#' dataMarginalTrend <- generate_data("marginalTrend", n = 50)
#'
#' # Perform two tailed MK test on sample data:
#' mkUnivariate(dataMarginalTrend[, 1])
#' mkUnivariate(dataMarginalTrend[, 2])
#'
#' @seealso [mkComponent]: The multivariate extension of this test.
#'
#' @importFrom stats qnorm pnorm
#' @export

mkUnivariate <- function(x) {}

mkUnivariate <- function(x) {

  # Check for data type
  if (!is.numeric(x)){
    stop("'x' must be a numeric vector.")
  }

  n = length(x)
  M = 0

  # Computation of the MK TS
  for (i in 1:(n-1)) {
    for (j in (i+1):n){
      M = M + sign(x[j] - x[i])
    }
  }

  # Computation of variance
  var_ = (n * (n - 1) * (2 * n + 5)) / 18

  # Standardized TS
  if(M > 0) {
    Z = (M-1)/sqrt(var_)
  } else if (M == 0) {
    Z = 0
  } else {Z = (M+1)/sqrt(var_)}

  # p-value
  pvalue = 2*pnorm(-abs(Z))

  results = list("statistic" = M, "p.value" = pvalue)

  return(results)
}



#===============================================================================

#' @title Component Wise Mann-Kendall Test Statistic
#'
#' @description The functions performs the univariate Mann-Kendall test statistic
#'  to each variable of a data series.
#'
#' @param data Numeric matrix representing the data series, each column should be a component.
#'
#' @return A numeric vector with the univariate MK test statistic for each component of the data series.
#'
#' @details
#' Let \eqn{M} be a dataset of \eqn{m} components and \eqn{n} observations.
#' The Mann-Kendall's (MK) test statistic for a variable of the dataset \eqn{M^{(u)}} is given by:
#'
#' \deqn{M^{(u)} = \sum_{1 \leq i \leq j \leq n} sgn(x_j^{(u)} - x_i^{(u)})}
#' where \eqn{sgn(\cdot)} is the sign function:
#' \deqn{
#' sgn(x)=\begin{cases}
#' -1 \quad \text{if } x<0, \\
#' 0 \quad \text{if } x=0, \\
#' +1 \quad \text{if } x>0
#' \end{cases}
#' }
#'
#' This test statistic is normal distributed, with mean and variance:
#' \deqn{E(M^{(u)}) = 0},
#' \deqn{\text{var}(M^{(u)}) = \frac{n(n-1)(2n+5)}{18}}
#'
#' @references
#'
#' - Hamed, K.H., Rao, A.R., 1998. A modified Mann-Kendall trend test for autocorrelated
#'    data. J. Hydrol. 204 (1–4), 182–196.
#' - Kendall, M., (1975). Rank Correlation Methods; Griffin: London.
#'
#' @examples
#' # Sample data (Both marginal distributions have trend):
#' dataMarginalTrend <- generate_data("marginalTrend", n = 50)
#'
#' # Perform multivariate MK test on sample data:
#' mkComponent(dataMarginalTrend)
#'
#' @export

mkComponent <- function(data) {}

mkComponent <- function(data) {

  data_ <- as.matrix(data)

  # Check for data type
  if(!is.numeric(data_)){
    stop("'data' must be a numeric matrix.")
  }

  size <- dim(data_)
  n <- size[1]

  M <- matrix(nrow = 1, ncol = size[2], byrow = TRUE)

  # Computation of the MK TS for each column
  for (u in 1:size[2]){
    M[u] <- 0
    for (i in 1:(n-1)){
      for (j in (i+1):n){
        M[u] <- M[u] + sign(data_[j, u] - data_[i, u])
      }
    }
  }

  return(M)
}
