#' QQ Plot with Correlation
#' @name QQnorm
#'
#' @description
#' Creates QQ plot of complete or censored data, with summaries and test.
#'
#' @usage
#' QQnorm(X, main="", ylab="", censor=0, winsor=0,
#'        joinem=FALSE, ylim=c(NA,NA), isBC=FALSE, is2pBC=FALSE,
#'        doplot=TRUE, showP=TRUE, fitline=TRUE, showsum=FALSE)
#'
#' @param X    the numeric data vector to be plotted.  Censored values should
#'   be reported as the censoring value.
#' @param main    the header text.
#' @param ylab    the name of the variable.
#' @param censor  *optional* (default of 0) - the number of data censored on the left.
#' @param winsor  *optional* (default of 0) - the number of data winsored in each tail.
#' @param joinem  *optional* (default of FALSE) - if TRUE the plot is drawn as a
#'   segmented line, if FALSE the individual points are plotted as x if winsorized, else *.
#' @param ylim    *optional* - y limits on the plot.
#' @param isBC    *optional* (default of FALSE) - if true, the data set is a Box-Cox
#'   transform of the original data.
#' @param is2pBC  *optional* (default of FALSE) - if true, the data set is a
#'   shifted Box-Cox tranform.
#' @param doplot  *optional* (default of TRUE) - if true, the QQ plot is drawn.
#' @param showP   *optional* (default of TRUE) - if true, the QQCC P value is shown on the plot.
#' @param fitline *optional* (default of TRUE) - if true, the QQ regression line is plotted.
#' @param showsum *optional* (default of FALSE) - if true, the intercept and
#'   slope of the QQ regression are shown.
#'
#' @details
#' Makes QQ plot of complete or censored data, as input by user:
#'    * fits regression line to complete uncensored portion of data,
#'    * calculates the QQ correlation coefficient of the fitted line, and
#'    * reports the P value of the QQ correlation coefficient as calculated by `BCR_pval`.
#' Parameters `isBC` and `is2pBC` are relevant to the P value calculation.
#'
#' @returns A list containing the following components:
#'
#'   \item{correl}{the QQ correlation coefficient.}
#'   \item{Pval}{the P value of the QQ correlation coefficient.}
#'   \item{mean}{the mean of the data.}
#'   \item{sd}{the sd of the data.}
#'   \item{intercept}{the intercept of the QQ regression line, used in place of
#'                    the mean when there is censoring or winsorization.}
#'   \item{slope}{the slope of the QQ regression line, used in place of
#'                the sd when there is censorin or winsorization.}
#'
#' @author Douglas M. Hawkins, Jessica J. Kraker <krakerjj@uwec.edu>
#'
#' @examples
#' # parameters
#' mu    <- 40
#' sigma <- 10
#' n     <- 120
#' # identifying winsoring
#' wins  <- trunc(n/40)
#' # replicable randomization
#' set.seed(1069)
#' X     <- mu + sigma*rnorm(n)
#' # replicable randomization with heavy tails
#' set.seed(1069)
#' HT    <- mu + sigma * rt(n, 5)
#'
#' # retain original visual settings
#' oldsettings <- par(mfrow=par()$mfrow, oma=par()$oma)
#' # visual settings
#' par(mfrow=c(2,2))
#' par(oma=c(0,0,2,0))
#' # plot to compare
#' base <- QQnorm(X, main="Base normal", showsum=TRUE)
#' title("Illustrating QQnorm with para_limits", outer=TRUE)
#' basew <- QQnorm(X, main="Winsorized", winsor=wins, showsum=TRUE)
#' ht   <- QQnorm(HT, main="Heavy tail", showsum=TRUE)
#' htw  <- QQnorm(HT, main="Winsorized", winsor=wins, showsum=TRUE)
#' # restore
#' par(oldsettings)
#'
#' @references Hawkins DM, Esquivel RN (2024). A Quantile-Quantile Toolbox for
#' Reference Intervals.  *The Journal of Applied Laboratory Medicine*, **9:2**, 357-370.
#'
#' @importFrom stats coef
#' @importFrom stats cor
#' @importFrom stats lm
#' @importFrom stats qnorm
#' @importFrom graphics legend
#' @importFrom graphics abline
#'
#' @export

QQnorm = function(X, main="", ylab="", censor=0, winsor=0,
                  joinem=FALSE, ylim=c(NA,NA), isBC=FALSE, is2pBC=FALSE, doplot=TRUE,
                  showP=TRUE, fitline=TRUE, showsum=FALSE) {
  mean    <- NA
  sd      <- NA
  if (censor + winsor == 0) {  # Don't get mean or sd if censored or winsorized
    mean    <- mean(X)
    sd      <- sd  (X)
  }
  winsor  <- trunc(winsor)
  n       <- length(X)
  pch     <- rep(1, n)
  if (winsor > 0) {
    pch[1:winsor] <- 4
    pch[(n-winsor+1):n] <- 4
  }
  ns      <- qnorm(((1:n)-0.5)/n)
  X    <- sort(X)
  range   <- (censor+1) : n
  QQrange <- (max(c(censor,winsor))+1) : (n-winsor)
  ltype   <- "p"
  if (joinem) ltype = "l"
  if (is.na(sum(ylim))) ylim <- c(min(X), max(X))
  correl  <- cor(ns[QQrange], X[QQrange])
  Pval    <- BCr_Pval(correl, n, censor, winsor, isBC, is2pBC)
  leg     <- paste("r=", round(correl,4))
  leg2    <- sprintf("P=%6.5f", Pval)
  linef   <- lm(X[QQrange] ~ ns[QQrange])
  if (doplot) {
    plot(ns[range], X[range], main=main, xlab="Normal scores",
         ylab=ylab, type=ltype, ylim=ylim, pch=pch)
    if (showP) leg <- paste(leg, leg2, sep="\n")
    legend("topleft", leg, bty="n")

    if (fitline) abline(linef)
    bl   <- ""
    summ <- sprintf("cut %4.4g\nslope  %4.4g",
                    coef(linef)[1], coef(linef)[2])
    if (showsum) bl <- paste(bl, summ, sep="")
    if (winsor > 0) {
      vline <- 0.5*(ns[winsor]+ns[winsor+1])
      abline(v= vline, lty=3)
      abline(v=-vline, lty=3)
    }
    legend("right", legend=bl, bty="n")
  }
  return(list(correl=correl, Pval=Pval, mean=mean, sd=sd,
              intercept=coef(linef)[1], slope=coef(linef)[2]))
}
