% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BC_Limits.r
\name{BC_limits}
\alias{BC_limits}
\title{Limits for Box-Cox Transformed Data}
\usage{
BC_limits(X, perc=0.95, cover=0.9, censor = 0, winsor=0,
          bottom=-3, top=3, epsilon=0.0001, neff=NA, CI_corrfac=NA,
          printem=FALSE)
}
\arguments{
\item{X}{the numeric data vector to be transformed.}

\item{perc}{\emph{optional} (default of 0.95) - the two-sided coverage of the
reference range computed.}

\item{cover}{\emph{optional} (default of 0.9) - the confidence level of the CI
computed for the reference limits.}

\item{censor}{\emph{optional} (default of 0) - the number of left-censored readings}

\item{winsor}{\emph{optional} (default of 0) - the number winsorised in each tail}

\item{bottom}{\emph{optional} (default of -3) - the smallest Box-Cox power to be considered.}

\item{top}{\emph{optional} (default of 3) - the largest Box-Cox power to be considered.}

\item{epsilon}{\emph{optional} (default of 0.0001) - a tolerance limit for convergence.}

\item{neff}{\emph{optional} (default of NA) - effective sample size, computed by
the code but can be overridden.}

\item{CI_corrfac}{\emph{optional} (default of NA) - correction factor for CIs,
computed by code but can be overridden.}

\item{printem}{\emph{optional} - if TRUE, routine will print out results as a \code{message}.}
}
\value{
A list containing the following components:

\item{bestr}{the maximized QQ correlation coefficient.}
\item{bestpow}{the fitted Box Cox power.}
\item{bestxform}{the fitted Box Cox transform of the data.}
\item{lower}{the lower reference limit and CI on the original scale.}
\item{upper}{the upper reference limit and CI on the original scale.}
\item{BClower}{the lower reference limits and CI on the transformed scale.}
\item{BCupper}{the upper reference limits and CI on the transformed scale.}
\item{meanof}{the mean of the Box-Cox transform.}
\item{sdof}{the sd of the Box-Cox transform.}
\item{intercept}{the intercept of the fitted QQ regression.}
\item{slope}{the slope of the fitted QQ regression.}
\item{Pval}{the P value of the QQ correlation.}
}
\description{
Box Cox transformation of data to normality; reference limits are found
using golden section search.
}
\details{
The function fits the Box-Cox transformation by finding the exponent that
maximizes the QQ correlation coefficient.
Having done so, it
\itemize{
\item calculates the reference limits and confidence intervals of the transformed data and
\item then transforms them back to the original scale.
The QQ analysis may incorporate censoring or winsorizing if appropriate.
}
}
\examples{
# parameters
mul    <- 3.6
sigmal <- 0.75

# replicable randomization
set.seed(1069)
X      <- exp(mul + sigmal*rnorm(120))

# evaluate and review
BC_results <- BC_limits(X, printem=TRUE)
BC_results$bestpow
BC_results$bestr
# original-scale limits
BC_results$lower[1]; BC_results$upper[1]
cat("\nWith 90\% [default] confidence, the lower limit is between",
    signif(BC_results$lower[2],5), "and", signif(BC_results$lower[3],5),
    ";\n while the upper limit is between",
    signif(BC_results$upper[2],5),"and",signif(BC_results$upper[3],5),".\n\n")

# adjust to have heavy tails
HT     <- X
HT[c(1,2,3,4)] <- HT[c(1,2,3,4)] * c(0.5, 0.5, 2, 2)

# evaluate and review
BC_HT_results <- BC_limits(HT)
BC_HT_results$lower; BC_HT_results$upper
# winsorized
BC_HT_wins_results <- BC_limits(HT, winsor=3)
BC_HT_wins_results$lower; BC_HT_wins_results$upper

}
\references{
Hawkins DM, Esquivel RN (2024). A Quantile-Quantile Toolbox for
Reference Intervals.  \emph{The Journal of Applied Laboratory Medicine}, \strong{9:2}, 357-370.
}
\author{
Douglas M. Hawkins, Jessica J. Kraker \href{mailto:krakerjj@uwec.edu}{krakerjj@uwec.edu}
}
