% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_MC_CW_OSL_DELOC.R
\encoding{UTF-8}
\name{run_MC_CW_OSL_DELOC}
\alias{run_MC_CW_OSL_DELOC}
\title{Run Monte-Carlo Simulation for CW-OSL (delocalized transitions)}
\usage{
run_MC_CW_OSL_DELOC(
  A,
  times,
  clusters = 10,
  N_e = 200,
  n_filled = N_e,
  R,
  method = "par",
  output = "signal",
  ...
)
}
\arguments{
\item{A}{\link{numeric} (\strong{required}): The optical excitation rate from trap to conduction band (s^-1)}

\item{times}{\link{numeric} (\strong{required}): The sequence of temperature steps within the simulation (s)}

\item{clusters}{\link{numeric} (\emph{with default}): The number of created clusters for the MC runs. The input can be the output of \link{create_ClusterSystem}. In that case \code{n_filled} indicate absolute numbers of a system.}

\item{N_e}{\link{integer} (\emph{with default}): The total number of electron traps available (dimensionless). Can be a vector of \code{length(clusters)}, shorter values are recycled.}

\item{n_filled}{\link{integer} (\emph{with default}): The number of filled electron traps at the beginning
of the simulation (dimensionless). Can be a vector of \code{length(clusters)}, shorter values are recycled.}

\item{R}{\link{numeric} (\strong{required}): The retrapping ratio for delocalized transitions (dimensionless)}

\item{method}{\link{character} (\emph{with default}): Sequential \code{'seq'} or parallel \code{'par'}processing. In
the parallel mode the function tries to run the simulation on multiple CPU cores (if available) with
a positive effect on the computation time.}

\item{output}{\link{character} (\emph{with default}): Output is either the \code{'signal'} (the default) or
\code{'remaining_e'} (the remaining charges, electrons, in the trap)}

\item{\dots}{further arguments, such as \code{cores} to control the number of used CPU cores or \code{verbose} to silence the terminal}
}
\value{
This function returns an object of class \code{RLumCarlo_Model_Output} which
is a \link{list} consisting of an \link{array} with dimension length(times) x clusters
and a \link{numeric} time vector.
}
\description{
Runs a Monte-Carlo (MC) simulation of continuous wave optically stimulated luminescence
(CW-OSL) using the one trap one recombination centre (OTOR) model.
The term delocalized here refers to the involvement of the conduction band.
}
\details{
\strong{The model}

\deqn{
I_{DELOC}(t) = -dn/dt = A * (n^2 / (N*R + n(1-R)))
}

Where in the function: \cr
t := time (s) \cr
A := the optical excitation rate from trap to conduction band (1/s)\cr
n := \code{n_filled}, the instantaneous number of electrons \cr
N := \code{N_e} the available number of electron traps available \cr
R := retrapping ratio for delocalized transitions
}
\section{Function version}{
 0.1.0
}

\examples{
## brief example
run_MC_CW_OSL_DELOC(
 A = 0.12,
 R = 0.1,
 times = 0:10,
 clusters = 10,
 method = "seq") \%>\%
plot_RLumCarlo(legend = TRUE)

## A long example
\dontrun{
A <- c(0.1,0.3,0.5,1)
times <- seq(0, 60, 1)
s <- 1e12
E <- 1
R <- c(1e-7, 1e-6, 0.01, 0.1) # sequence of different R values
clusters <- 1000 # number of Monte Carlo simulations
N_e <- c(200, 500, 700, 400) # number of free electrons
n_filled <- c(200, 500, 100, 70) # number of filled traps
method <-"par"
output <- "signal"
col <- c(1,2,3,4) # ifferent colours for the individual curves
plot_uncertainty <- c(TRUE,FALSE,TRUE,FALSE)  # do you want to see the uncertainty?
add_TF <- c(FALSE,rep(TRUE, (length(R)-1)))

## loop to plot different curves into one plot
for (u in 1:length(R)){
 results <- run_MC_CW_OSL_DELOC(
  A = A[u],
  times,
  clusters = clusters,
  N_e = N_e[u],
  n_filled = n_filled[u],
  R = R[u],
  method = method,
  output = output)

plot_RLumCarlo(
results,
 add = add_TF[u],
 legend = FALSE,
 col = col[u],
 main = "Delocalised Transition")
}
# add your legend with your parameters
legend("topright",
  ncol = 4,
  cex = 0.55,
  title = "parameters",
  legend=c(
   paste0("A = ", A),
   paste0("n_filled = ", n_filled),
   paste0("N_e = ", N_e),
   paste0("R = ", R)),
   bty = "n",
   text.col = col)
}

} 

\section{How to cite}{
Kreutzer, S., 2025. run_MC_CW_OSL_DELOC(): Run Monte-Carlo Simulation for CW-OSL (delocalized transitions). Function version 0.1.0. In: Friedrich, J., Kreutzer, S., Pagonis, V., Schmidt, C., 2025. RLumCarlo: Monte-Carlo Methods for Simulating Luminescence Phenomena. R package version 0.1.10. https://r-lum.github.io/RLumCarlo/
}

\references{
Pagonis, V., Friedrich, J., Discher, M., Müller-Kirschbaum, A., Schlosser, V., Kreutzer, S.,
Chen, R. and Schmidt, C., 2019. Excited state luminescence signals from a random
distribution of defects: A new Monte Carlo simulation approach for feldspar.
Journal of Luminescence 207, 266–272. \doi{10.1016/j.jlumin.2018.11.024}

\strong{Further reading}

Chen, R., McKeever, S.W.S., 1997. Theory of Thermoluminescence and Related Phenomena.
WORLD SCIENTIFIC. \doi{10.1142/2781}
}
\author{
Sebastian Kreutzer, Institute of Geography, Heidelberg University (Germany)
}
\keyword{data}
\keyword{models}
