% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rvn_rvt_write_met.R
\name{rvn_rvt_write_met}
\alias{rvn_rvt_write_met}
\title{EC Climate Station File Conversion}
\usage{
rvn_rvt_write_met(
  metdata,
  rvt_met_mapping = NULL,
  filenames = NULL,
  met_file_prefix = "met_",
  prd = NULL,
  write_stndata = TRUE,
  filename_stndata = "met_stndata.rvt",
  NA_value = -1.2345
)
}
\arguments{
\item{metdata}{EC meteorological data from one or more stations (e.g., from \code{weathercan::weather_dl()})}

\item{rvt_met_mapping}{list that provides the mapping between metdata names and those used in Raven}

\item{filenames}{(optional) character vector of filenames for the rvt data files, length same as number of stations in metdata}

\item{met_file_prefix}{(optional) prefixes the file name (default: "met_")}

\item{prd}{(optional) data period to use in .rvt file}

\item{write_stndata}{(optional) write the gauge data to a separate .rvt file}

\item{filename_stndata}{(optional) name of the station data file created (if \code{write_stndata=TRUE})}

\item{NA_value}{(optional) value to use for NA values in rvt file (default -1.2345 for Raven format)}
}
\value{
\item{TRUE}{return \code{TRUE} if the function is executed properly}
}
\description{
Converts meteorological data for a given station into the
Raven .rvt format.
}
\details{
Writes data in either :Data or :MultiData format depending on the number of
supported parameter columns provided. The data should be downloaded and prepared
with missing days included, and preferably is downloaded directly using \code{weathercan}.

metdata contains all of the meteorological data to be written to file. metdata should be
in a tibble or data frame format, and is required to have a date/time column (either DATE,
TIME, or DATETIME, all of which have at least a Date component), the STATION_NAME,
and all desired forcings to be written to file. If the columns ELEV, LAT, LON are included,
the station meta data may be written to a separate rvt file as well (see below). All supported
data columns in metdata are written into rvt format, so the desired columns
for the rvt file should be passed through metdata and filtered first if needed.

rvt_met_mapping is a list that maps the metdata column names to Raven variables. If
\code{weathercan} is used then this may be left NULL, as the mapping is automatically provided.
Otherwise, the user must convert all desired column names to Raven-recognized names, or
provide the mapping information as a list through this parameter. An example format can be seen
(the mapping used for weathercan by default) in \code{data("rvn_rvt_mappings_data")}.

filenames may be used to provide the specific desired filenames (with paths) for each
station rvt data file being generated; this should be a character vector of length
equal to the number of unique station names in the data.

Note that the function uses
\code{sort(unique(metdata$STATION_NAME))} to determine the order of stations,
thus the filenames should correspond to the sorted vector of station numbers as well.

prd is used by the xts formatted-data to restrict the data reported in .rvt
files, for each station, to this period. The prd should be defined in
"YYYY-MM-DD/YYYY-MM-DD" string format. If the period supplied results in an
empty time series (i.e. non-overlapping time periods), an error will be
thrown.

met_file_prefix can be used to add a prefix to the .rvt data file names, ("met_" by default)
which may be useful in organizing multiple climate data files. This is ignored
if filenames are specified.

write_stndata will print out the gauge(s) metadata to file (specified by
filename_stndata parameter) in the .rvt format, which is required to include a
meterological station in Raven. The function will append to the file if it already exists, meaning
that this works for iterations of this function. metdata must include the columns
ELEV, LAT, and LON if station data is to be written, else the meta data file will not be created.

The function has several built-in data quality checks. These include:

* checking that the time interval is consistent for each station;
* ensuring that meta data is unique for each station name; and
* check for missing data and issuing a warning that post-processing will be required

Data quality is not assessed in this package, such as consistency between
minimum and maximum temperatures and missing data. Consider viewing the RavenR.extras package for
functions to interpolate missing meteorological data and checking for min/max temperature consistency.

This function is designated to use data from the weathercan package, but may be used with any
supplied data frame of meteorological information if in the correct format and other
relevant information (such as rvt_met_mapping, if needed) is supplied. The
weathercan package is external to RavenR and is not an explicit dependency
of RavenR, although a sample weathercan data set can be viewed as \code{data(rvn_weathercan_sample)}.
}
\examples{
# note: example modified to avoid using weathercan directly, uses saved
## weathercan data from RavenR package sample data
data(rvn_weathercan_sample)
kam <- rvn_weathercan_sample

# basic use, override filename to temporary file
rvn_rvt_write_met(metdata = kam,
  filenames = file.path(tempdir(), "rvn_rvt_metfile.rvt"),
  filename_stndata = file.path(tempdir(), "met_stndata.rvt"))

}
\seealso{
\code{\link{rvn_rvt_write}} to write non-forcing time series data to Raven rvt format.

Download Environment Canada Historical weather data from (climate.weather.gc.ca), or use the
`weathercan` package to access this data through R.
}
