#' Estimate Squared Partial Correlation Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of squared partial correlation coefficients
#'   (\eqn{r^{2}_{p}})
#'   is derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{r^{2}_{p}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{args}{Function arguments.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{r^{2}_{p}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{r^{2}_{p}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{r^{2}_{p}}.}
#'     \item{fun}{Function used ("PCorMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#' @examples
#' # Data ---------------------------------------------------------------------
#' data("nas1982", package = "betaMC")
#'
#' # Fit Model in lm ----------------------------------------------------------
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#'
#' # MC -----------------------------------------------------------------------
#' mc <- MC(
#'   object,
#'   R = 100, # use a large value e.g., 20000L for actual research
#'   seed = 0508
#' )
#'
#' # PCorMC -------------------------------------------------------------------
#' out <- PCorMC(mc, alpha = 0.05)
#'
#' ## Methods -----------------------------------------------------------------
#' print(out)
#' summary(out)
#' coef(out)
#' vcov(out)
#' confint(out, level = 0.95)
#'
#' @family Beta Monte Carlo Functions
#' @keywords betaMC pcor
#' @export
PCorMC <- function(object,
                   alpha = c(0.05, 0.01, 0.001)) {
  stopifnot(
    inherits(
      x = object,
      what = "mc"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  if (object$fun == "MCMI") {
    est <- colMeans(
      do.call(
        what = "rbind",
        args = lapply(
          X = object$args$mi_output$lm_process,
          FUN = function(x) {
            x$pcor
          }
        )
      )
    )
  } else {
    est <- object$lm_process$pcor
  }
  sr <- SCorMC(object)
  sr <- sr$thetahatstar
  rsq <- RSqMC(object)
  rsq <- rsq$thetahatstar
  thetahatstar <- mapply(
    sr = sr,
    rsq = rsq,
    SIMPLIFY = FALSE,
    FUN = function(sr,
                   rsq) {
      .PCorSq(
        srsq = sr^2,
        rsq = rsq[1]
      )
    }
  )
  out <- list(
    call = match.call(),
    args = list(
      object = object,
      alpha = alpha
    ),
    thetahatstar = thetahatstar,
    est = est,
    fun = "PCorMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  out
}
