#' Estimate Multiple Correlation Coefficients
#' (R-Squared and Adjusted R-Squared)
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details R-squared (\eqn{R^{2}}) and
#'   adjusted R-squared (\eqn{\bar{R}^{2}})
#'   are derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{R^{2}} and \eqn{\bar{R}^{2}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{args}{Function arguments.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{R^{2}} and \eqn{\bar{R}^{2}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{R^{2}} and \eqn{\bar{R}^{2}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{R^{2}} and \eqn{\bar{R}^{2}}.}
#'     \item{fun}{Function used ("RSqMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#' @examples
#' # Data ---------------------------------------------------------------------
#' data("nas1982", package = "betaMC")
#'
#' # Fit Model in lm ----------------------------------------------------------
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#'
#' # MC -----------------------------------------------------------------------
#' mc <- MC(
#'   object,
#'   R = 100, # use a large value e.g., 20000L for actual research
#'   seed = 0508
#' )
#'
#' # RSqMC --------------------------------------------------------------------
#' out <- RSqMC(mc, alpha = 0.05)
#'
#' ## Methods -----------------------------------------------------------------
#' print(out)
#' summary(out)
#' coef(out)
#' vcov(out)
#' confint(out, level = 0.95)
#'
#' @family Beta Monte Carlo Functions
#' @keywords betaMC rsq
#' @export
RSqMC <- function(object,
                  alpha = c(0.05, 0.01, 0.001)) {
  stopifnot(
    inherits(
      x = object,
      what = "mc"
    )
  )
  if (object$fun == "MCMI") {
    est <- colMeans(
      do.call(
        what = "rbind",
        args = lapply(
          X = object$args$mi_output$lm_process,
          FUN = function(x) {
            x$rsq
          }
        )
      )
    )
  } else {
    est <- object$lm_process$rsq
  }
  thetahatstar <- lapply(
    X = object$thetahatstar,
    FUN = function(x) {
      rsq <- .RSqofSigma(
        sigmacap = x$sigmacap,
        k = object$lm_process$k
      )
      adj <- (
        1 - (
          1 - rsq
        ) * (
          (
            object$lm_process$n - 1
          ) / object$lm_process$df
        )
      )
      c(rsq = rsq, adj = adj)
    }
  )
  out <- list(
    call = match.call(),
    args = list(
      object = object,
      alpha = alpha
    ),
    thetahatstar = thetahatstar,
    est = est,
    fun = "RSqMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  out
}
