% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lav_betaselect.R
\name{lav_betaselect}
\alias{lav_betaselect}
\title{Betas-Select in a 'lavaan'-Model}
\usage{
lav_betaselect(
  object,
  to_standardize = ".all.",
  not_to_standardize = NULL,
  skip_categorical_x = TRUE,
  output = c("data.frame", "text"),
  std_se = c("none", "delta", "bootstrap"),
  std_z = TRUE,
  std_pvalue = TRUE,
  std_ci = TRUE,
  level = 0.95,
  progress = TRUE,
  boot_out = NULL,
  bootstrap = 100L,
  store_boot_est = TRUE,
  parallel = c("no", "snow", "multicore"),
  ncpus = parallel::detectCores(logical = FALSE) - 1,
  cl = NULL,
  iseed = NULL,
  find_product_terms = TRUE,
  check_mean_centering = FALSE,
  std_intercept = FALSE,
  ...,
  delta_method = c("lavaan", "numDeriv"),
  vector_form = TRUE
)
}
\arguments{
\item{object}{The output of
\code{lavaan} model fit functions, such
as \code{\link[lavaan:sem]{lavaan::sem()}} and \code{\link[lavaan:cfa]{lavaan::cfa()}}.}

\item{to_standardize}{A string vector,
which should be the names of the
variables to be standardized.
Default is \code{".all."}, indicating all
variables are to be standardized
(but see \code{skip_categorical_x}).}

\item{not_to_standardize}{A string
vector, which should be the names
of the variables that should not be
standardized. This argument is useful
when most variables, except for a few,
are to be standardized. This argument
cannot be ued with \code{to_standardize}
at the same time. Default is \code{NULL},
and only \code{to_standardize} is used.}

\item{skip_categorical_x}{Logical.
If \code{TRUE}, the default, all
categorical predictors, defined as
variables with only two possible
values in the data analyzed, will
be skipped in standardization. This
overrides the argument
\code{to_standardize}. That is, a
categorical predictor will not be
standardized even if listed in
\code{to_standardize}, unless users set
this argument to \code{FALSE}.}

\item{output}{The format of the
output. Not used because the format
of the printout is now controlled
by the \code{print}-method of the output
of this function. Kept for backward
compatibility.}

\item{std_se}{String. If set to \code{"none"},
the default, standard errors will not
be computed for the standardized
solution. If set to \code{"delta"},
delta method will be used to compute
the standard errors. If set to
\code{"bootstrap"}, then what it does
depends whether \code{boot_out} is set.
If \code{boot_out} is to an output of
\code{\link[manymome:do_boot]{manymome::do_boot()}}, its content
will be used. If \code{boot_out} is
\code{NULL} \emph{and} bootstrap
estimates are available in \code{object}
(e.g., bootstrapping is requested
when fitting the model in \code{lavaan}),
then the stored bootstrap estimates
will be sued. If not available,
the bootstrapping will be conducted
using \code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}},
using arguments \code{bootstrap},
\code{parallel}, \code{ncpus}, \code{cl}, and
\code{iseed}.`}

\item{std_z}{Logical. If \code{TRUE} and
\code{std_se} is not set to \code{"none"},
standard error will be computed
using the method specified in
\code{std_se}. Default is \code{TRUE}.}

\item{std_pvalue}{Logical. If \code{TRUE},
\code{std_se} is not set to \code{"none"},
and \code{std_z} is \code{TRUE}, \emph{p}-values
will be computed using the method
specified in \code{std_se}. For
bootstrapping, the method proposed by
Asparouhov and Muthén (2021) is used.
Default is \code{TRUE}.}

\item{std_ci}{Logical. If \code{TRUE} and
\code{std_se} is not set to \code{"none"},
confidence intervals will be
computed using the method specified in
\code{std_se}. Default is \code{FALSE.}}

\item{level}{The level of confidence
of the confidence intervals. Default
is .95. It will be used in the
confidence intervals of both
the unstandardized and
standardized solution.}

\item{progress}{Logical. If \code{TRUE},
progress bars will be displayed
for long process.}

\item{boot_out}{If \code{std_se} is
\code{"bootstrap"} and this argument
is set to an output of
\code{\link[manymome:do_boot]{manymome::do_boot()}}, its output
will be used in computing statistics
such as standard errors and
confidence intervals. This allows
users to use methods other than
bootstrapping when fitting the
model, while they can still request
bootstrapping for the standardized
solution.}

\item{bootstrap}{If \code{std_se} is
\code{"bootstrap"} but bootstrapping is
not requested when fitting the model
and \code{boot_out} is not set,
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}} will be
called to do bootstrapping. This
argument is the number of bootstrap
samples to draw. Default is 100.
Should be set to 5000 or even 10000
for stable results.}

\item{store_boot_est}{Logical. If
\code{std_se} is \code{"bootstrap"} and this
argument is \code{TRUE}, the default,
the bootstrap estimates of the
standardized solution will be stored
in the attribute \code{"boot_est"}. These
estimates can be used for
diagnosis of the bootstrapping. If
\code{FALSE}, then the bootstrap estimates
will not be stored.}

\item{parallel}{If \code{std_se} is
\code{"bootstrap"} but bootstrapping is
not requested when fitting the model
and \code{boot_out} is not set,
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}} will be
called to do bootstrapping. This
argument is to be passed to
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}. Default
is \code{"no"}.}

\item{ncpus}{If \code{std_se} is
\code{"bootstrap"} but bootstrapping is
not requested when fitting the model
and \code{boot_out} is not set,
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}} will be
called to do bootstrapping. This
argument is to be passed to
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}. Default
is \code{parallel::detectCores(logical = FALSE) - 1}.
Ignored if \code{parallel} is \code{"no"}.}

\item{cl}{If \code{std_se} is
\code{"bootstrap"} but bootstrapping is
not requested when fitting the model
and \code{boot_out} is not set,
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}} will be
called to do bootstrapping. This
argument is to be passed to
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}}. Default
is \code{NULL}.
Ignored if \code{parallel} is \code{"no"}.}

\item{iseed}{If \code{std_se} is
\code{"bootstrap"} but bootstrapping is
not requested when fitting the model
and \code{boot_out} is not set,
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}} will be
called to do bootstrapping. This
argument is to be passed to
\code{\link[lavaan:bootstrap]{lavaan::bootstrapLavaan()}} to set
the seed for the random resampling.
Default
is \code{NULL}. Should be set to an integer
for reproducible results.
Ignored if \code{parallel} is \code{"no"}.}

\item{find_product_terms}{String.
If it is certain that a model does
not have product terms, setting this
to \code{FALSE} will skip the search, which
is time consuming for a models with
many paths and/or many variables.
Default is \code{TRUE}, and the function
will automatically identify product
terms, if any.}

\item{check_mean_centering}{Logical.
If \code{TRUE}, it will check whether
variables involved in a product term
has been mean-centered. If not,
an error will be raised.}

\item{std_intercept}{Logical.
If \code{TRUE}, intercepts of \code{y} variables
will also be computed based on
the variables standardized.}

\item{...}{Optional arguments to be
passed to the \code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}},
which will be use to generate the
output.}

\item{delta_method}{The method used
to compute delta-method standard
errors. For internal use and should
not be changed.}

\item{vector_form}{The internal
method used to compute standardized
solution. For internal use and should
not be changed.}
}
\value{
A \code{lav_betaselect}-class object,
which is a data frame storing the parameter
estimates, similar in form to the
output of \code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}}.
}
\description{
Can standardize selected
variables in a \code{lavaan} model without
refitting the models, can handle
product term correctly and skip
categorical predictors in
standardization.
}
\details{
This function lets users
select which variables to be standardized
when computing the standardized
solution. It has the following
features:
\itemize{
\item It automatically skips predictors
which has only two unique values,
assuming that they are dummy variables.
\item It does not standardize product
term, which is incorrect. Instead,
it computes the product term with
its component variables standardized
first.
\item It can be used to generate bootstrap
confidence intervals for the
standardized solution (Falk, 2018). Bootstrap
confidence interval is better than
doing standardization \emph{before} fitting
a model because it correctly takes
into account the sampling variance
of the standard deviations. It is
also better than delta-method
confidence interval because it takes
into account the usually asymmetric
distribution of parameters after
standardization, such as standardized
loadings and correlations.
\item For comparison, it can also report
delta-method standard errors and
confidence intervals if requested.
}
\subsection{Problems With Common Approaches}{

In most SEM programs, users
have limited control on which
variables to standardize when
requesting the standardized solution.
The solution may be uninterpretable
or misleading in these conditions:
\itemize{
\item Dummy variables are standardized
and their coefficients cannot be interpreted as the
difference between two groups on the
outcome variables.
\item Product terms (interaction terms)
are standardized and they cannot be
interpreted as the changes in the
effects of focal variables when the
moderators change (Cheung, Cheung,
Lau, Hui, & Vong, 2022).
\item Variables with meaningful units can
be more difficult to interpret when
they are standardized (e.g., age).
}

Moreover, the delta method is usually
used in standardization, which is suboptimal for
standardization unless the sample
size is large (Falk, 2018). For example, the
covariance with variables standardized
is a correlation, and its sampling
distribution is skewed unless its
population value is zero. However,
delta-method confidence interval
for the correlation is necessarily
symmetric around the point estimate.
}

\subsection{Limitations}{
\itemize{
\item It only supports observed variable
interaction terms, and only support
two-way interactions.
\item It does not support multilevel
models.
\item It only supports models fitted to
raw data.
}
}
}
\examples{

library(lavaan)
# Need to mean-center iv and mod
data_test_medmod$iv <- data_test_medmod$iv - mean(data_test_medmod$iv)
data_test_medmod$mod <- data_test_medmod$mod - mean(data_test_medmod$mod)
mod <-
"
med ~ iv + mod + iv:mod
dv ~ med + iv
"
fit <- sem(mod,
           data_test_medmod,
           fixed.x = TRUE)
summary(fit)
fit_beta <- lav_betaselect(fit,
                           to_standardize = c("iv", "dv"))
fit_beta
print(fit_beta, standardized_only = FALSE)

# In real studies:
# - should set bootstrap to at least 5000
# - should set parallel to "snow" or "multicore"
fit_beta_boot <- lav_betaselect(fit,
                                to_standardize = c("iv", "dv"),
                                std_se = "bootstrap",
                                std_ci = TRUE,
                                bootstrap = 100,
                                iseed = 1234)
fit_beta_boot
print(fit_beta_boot, standardized_only = FALSE)

# Print full results
print(fit_beta_boot,
      standardized_only = FALSE)


}
\references{
Asparouhov, A., & Muthén, B. (2021). Bootstrap p-value computation.
Retrieved from https://www.statmodel.com/download/FAQ-Bootstrap\%20-\%20Pvalue.pdf

Cheung, S. F., Cheung, S.-H., Lau, E. Y. Y., Hui, C. H., & Vong, W. N.
(2022) Improving an old way to measure moderation effect in standardized
units. \emph{Health Psychology}, \emph{41}(7), 502-505.
\doi{10.1037/hea0001188}

Falk, C. F. (2018). Are robust standard errors the best approach
for interval estimation with nonnormal data in structural equation
modeling?
\emph{Structural Equation Modeling: A Multidisciplinary Journal, 25}(2)
244-266. \doi{10.1080/10705511.2017.1367254}
}
\seealso{
\code{\link[=print.lav_betaselect]{print.lav_betaselect()}} for its print method.
}
\author{
Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}
}
