#' Generate efficient 2-Level Fractional Factorial Designs Using Beam Search
#'
#' This function constructs efficient 2-level unblocked and blocked fractional factorial designs
#' using a beam–search–based generator selection algorithm.
#' It evaluates candidate generators using proxy criteria (K3, K4) and
#' returns the top-ranked designs along with generators, scores, alias
#' structure, and blocked designs.
#'
#' The function automatically:
#' \itemize{
#'   \item enumerates all possible generator masks,
#'   \item performs canonical ordering to avoid duplicates,
#'   \item evaluates designs using moment-based proxy criteria,
#'   \item selects best designs based on beam search,
#'   \item prints clean summaries (rank, design matrix, aliasing, blocks),
#'   \item returns a structured list of final designs.
#' }
#'
#' @param n Integer. Total number of factors (base factors + generators).
#' @param k Integer. Number of dependent generator columns to add i.e. size of the fraction.
#'   The resulting design has \code{r = n - k} base factors.
#' @param max_results Integer. Maximum number of final best-ranked designs
#'   to return. Default is 20.
#' @param beam_width Integer. Maximum beam width used in beam search.
#'   Default is 3000.
#' @param verbose Logical. If \code{TRUE}, prints detailed output for every
#'   ranked design, including alias structure and blocked design. Default is TRUE.
#'
#' @return A list (invisible) of the best-ranked designs.
#' Each element contains:
#' \itemize{
#'   \item \code{generators_str}: Defining contrast used for generation of design,
#'   \item \code{design}: final design matrix with renamed factor labels,
#'   \item \code{score}: proxy measures (K3, K4),
#'   \item \code{key}: canonical key used for uniqueness.
#' }
#'
#' @details
#' The function internally uses:
#' \itemize{
#'   \item binary encoding of columns for canonical keys,
#'   \item moment-based proxies K3 and K4,
#'   \item mask enumeration for generator creation,
#'   \item alias structure detection for main and 2-factor effects,
#'   \item automatic block generator selection (2-level).
#' }
#'
#' The generated output provides experimenters with statistically efficient
#' two-level fractional factorial designs that are well suited for both industrial
#' and agricultural research. By reducing the total number of experimental runs
#' while preserving the ability to estimate key main effects and low-order
#' interactions, these designs offer a resource-efficient framework for screening
#' factors, optimizing processes, and evaluating system performance under practical
#' field or laboratory constraints.
#'
#'
#' @examples
#' \donttest{
#' # Generate 2-level fractional factorial designs:
#' res <- dol2(n = 5, k = 2, max_results = 5, beam_width = 3000, verbose = TRUE)
#'
#' # Access first ranked design
#' res[[1]]$design
#' res[[1]]$generators_str
#' }
#'
#' @export
#' @references
#' Dash, S., Parsad, R. and Gupta, V. K. (2013).
#' Row–column Designs for 2^n factorial 2-Colour Microarray Experiments for
#' Estimation of Main Effects and Two-Factor Interactions with Orthogonal
#' Parameterization. *Agricultural Research*, 2(2), 172–182.
#'
#' National Bureau of Standards (1957).
#' *Fractional Factorial Experiment Designs for Factors at Two Levels*.
#' Applied Mathematics Series 48. US Government Printing Office, Washington DC.

dol2 <- function(n, k, max_results = 20,beam_width = 3000,  verbose = TRUE) {
modn <- function(x, n) { ((x %% n) + n) %% n }

  make_factor_names <- function(m) {
    letters_vec <- LETTERS
    if (m <= length(letters_vec)) return(letters_vec[1:m])
    res <- c(); i <- 1
    while (length(res) < m) {
      if (i == 1) res <- c(res, letters_vec) else for (a in letters_vec) res <- c(res, paste0(letters_vec[i %% length(letters_vec)], a))
      i <- i + 1
    }
    res[1:m]
  }

  gen_2level_from_base <- function(r) {
    if (r <= 0) return(data.frame())
    grid <- expand.grid(rep(list(0:1), r))
    names(grid) <- make_factor_names(r)
    grid
  }

  add_dependent_col_from_mask <- function(design_df, mask, newname) {
    if (length(mask) != ncol(design_df)) stop("mask length must match number of existing columns")
    vals <- as.matrix(design_df) %*% matrix(as.integer(mask), ncol = 1)
    design_df[[newname]] <- as.integer(vals %% 2)
    design_df
  }

  enumerate_masks_dynamic <- function(p) {
    ms <- vector("list", 0)
    if (p == 0) return(ms)
    lim <- 2^p - 1
    for (i in 1:lim) ms[[length(ms) + 1]] <- as.integer(intToBits(i))[1:p]
    ms
  }

  enumerate_masks_dynamic_simple <- function(p) {
    ms <- list()
    if (p == 0) return(ms)
    for (i in 1:(2^p - 1)) ms[[length(ms) + 1]] <- as.integer(intToBits(i))[1:p]
    ms
  }

  col_code_binary <- function(vec) {
    v <- as.integer(vec)
    if (all(v %in% c(-1L, 1L))) v <- as.integer((v == 1L))
    bits <- paste(rev(v), collapse = "")
    strtoi(bits, base = 2)
  }

  design_canonical_key_binary <- function(design_df) {
    M <- as.matrix(design_df)
    cols <- apply(M, 2, col_code_binary)
    paste(sort(cols), collapse = "_")
  }

  pairwise_delta_counts <- function(design) {
    N <- nrow(design)
    mat <- as.matrix(design)
    res <- matrix(0, nrow = N, ncol = N)
    for (i in 1:N) {
      eqs <- t(t(mat) == mat[i, ])
      res[i, ] <- rowSums(eqs)
    }
    res
  }

  compute_Kt <- function(design, t = 3) {
    N <- nrow(design)
    deltas <- pairwise_delta_counts(design)
    sum(deltas^t) / (N^2)
  }

  compute_Aproxy_2level <- function(design) {
    K3 <- compute_Kt(design, 3)
    K4 <- compute_Kt(design, 4)
    c(K3 = K3, K4 = K4)
  }

  mask2_to_string <- function(mask) {
    base_names <- make_factor_names(length(mask))
    idx <- which(as.integer(mask) == 1)
    if (length(idx) == 0) return("1")
    paste(base_names[idx], collapse = "*")
  }

  # Alias structure for 2-level (main & 2-factor)
  list_alias_structure_2level <- function(design_df, effects = c("main","2factor")) {
    M <- as.matrix(design_df)
    if (all(M %in% c(-1,1))) M <- (M == 1) * 1L
    p <- ncol(M); fac_names <- colnames(design_df)
    effect_list <- list(); effect_labels <- c()
    if ("main" %in% effects) for (i in 1:p) { effect_list[[length(effect_list)+1]] <- M[, i]; effect_labels <- c(effect_labels, fac_names[i]) }
    if ("2factor" %in% effects && p >= 2) {
      combs <- combn(p, 2, simplify = FALSE)
      for (pair in combs) {
        eff <- (M[, pair[1]] + M[, pair[2]]) %% 2
        effect_list[[length(effect_list)+1]] <- eff
        effect_labels <- c(effect_labels, paste(fac_names[pair], collapse=":"))
      }
    }
    if (length(effect_list) == 0) return(list())
    E <- do.call(cbind, effect_list); nc <- ncol(E); alias_groups <- list(); used <- rep(FALSE, nc)
    for (i in seq_len(nc)) {
      if (used[i]) next
      group <- i
      for (j in (i+1):nc) {
        if (j <= nc && all(E[, i] == E[, j])) { group <- c(group, j); used[j] <- TRUE }
      }
      if (length(group) > 1) alias_groups[[length(alias_groups)+1]] <- effect_labels[group]
    }
    alias_groups
  }

  # Block assignment (2-level) returns full with .block and list of blocks
  assign_blocks_2level <- function(design_df, mask) {
    M <- as.matrix(design_df)
    if (ncol(M) != length(mask)) stop("mask length must equal number of columns in design")
    block_id <- as.integer((M %*% matrix(as.integer(mask), ncol=1)) %% 2)
    block_label <- factor(block_id + 1, levels = 1:2, labels = paste0("block", 1:2))
    res <- design_df
    res$.block <- block_label
    blocks <- split(res[ , setdiff(names(res), ".block"), drop=FALSE], res$.block)
    list(full = res, blocks = blocks, block_id = block_label)
  }

  assess_confounding_2level <- function(design_df, mask, report = TRUE) {
    out_assign <- assign_blocks_2level(design_df, mask)
    full <- out_assign$full
    m <- ncol(design_df)
    main_confounded <- character(0); two_confounded <- character(0)
    for (j in seq_len(m)) {
      tab <- table(full$.block, full[[j]])
      balanced <- all(apply(tab, 2, function(col) length(unique(as.numeric(col))) == 1))
      if (!balanced) main_confounded <- c(main_confounded, colnames(full)[j])
    }
    if (m >= 2) {
      cols <- combn(seq_len(m), 2, simplify = FALSE)
      for (pp in cols) {
        i <- pp[1]; j <- pp[2]
        cells <- paste0(full[[i]], "_", full[[j]])
        tab <- table(full$.block, cells)
        balanced <- all(apply(tab, 2, function(col) length(unique(as.numeric(col))) == 1))
        if (!balanced) two_confounded <- c(two_confounded, paste0(colnames(full)[i], ":", colnames(full)[j]))
      }
    }
    res <- list(mask = mask,
                n_main_conf = length(main_confounded),
                n_two_conf = length(two_confounded),
                main_confounded = main_confounded,
                two_confounded = two_confounded)
    if (report) cat("mask:", paste(mask, collapse=""), "-> main_conf:", res$n_main_conf, " two_conf:", res$n_two_conf, "\n")
    res
  }

  auto_choose_block_generator_2level <- function(design_df, top_k = 3, verbose = TRUE) {
    p <- ncol(design_df)
    cands <- enumerate_masks_dynamic_simple(p)
    assessed <- lapply(cands, function(m) assess_confounding_2level(design_df, m, report = FALSE))
    summary_df <- do.call(rbind, lapply(assessed, function(x) {
      data.frame(mask = paste(x$mask, collapse=""), n_main_conf = x$n_main_conf, n_two_conf = x$n_two_conf, stringsAsFactors=FALSE)
    }))
    ord <- order(summary_df$n_main_conf, summary_df$n_two_conf)
    summary_df <- summary_df[ord, , drop = FALSE]
    top <- summary_df[1:min(top_k, nrow(summary_df)), , drop = FALSE]
    if (verbose) {
      cat("Top block generator candidates (2-level):\n"); print(top, row.names = FALSE)
    }
    list(summary = summary_df, top = top, top_assessed = assessed[ord[1:min(top_k, nrow(summary_df))]])
  }

  make_blocked_design_2level <- function(design_df, mask = NULL, auto = TRUE, top_k = 3, verbose = TRUE) {
    if (!is.null(mask) && length(mask) != ncol(design_df)) stop("mask length mismatch")
    if (auto) {
      cand <- auto_choose_block_generator_2level(design_df, top_k = top_k, verbose = FALSE)
      best <- cand$top_assessed[[1]]$mask
      chosen <- best
      chosen_report <- cand$top_assessed[[1]]
    } else {
      chosen <- mask
      chosen_report <- assess_confounding_2level(design_df, chosen, report = verbose)
    }
    assigned <- assign_blocks_2level(design_df, chosen)$blocks
    if (verbose) {
      for (bname in names(assigned)) {
        cat("\n=== ", bname, " (", nrow(assigned[[bname]]), "runs) ===\n", sep = "")
        print(assigned[[bname]])
      }
      cat("\nConfounding Summary:\n")
      cat("Mask: ", paste(chosen, collapse=" "), "\n")
      cat("Main Confounded: ",
          if (length(chosen_report$main_confounded) == 0) "None"
          else paste(chosen_report$main_confounded, collapse=", "), "\n")
      cat("Two-Factor Confounded: ",
          if (length(chosen_report$two_confounded) == 0) "None"
          else paste(chosen_report$two_confounded, collapse=", "), "\n")
    }
    list(blocks = assigned,
         confounding = list(mask = chosen, main_confounded = chosen_report$main_confounded, two_confounded = chosen_report$two_confounded))
  }

  rename_generated_columns <- function(design_df, r) {
    p <- ncol(design_df)
    new_names <- make_factor_names(p)
    if (!is.null(colnames(design_df))) {
      curr <- colnames(design_df)
      gidx <- grepl("^G\\d+$", curr)
      if (any(gidx)) { repl <- make_factor_names(p); curr[gidx] <- repl[gidx]; colnames(design_df) <- curr; return(design_df) }
    }
    colnames(design_df) <- new_names
    design_df
  }

  print_clean_two_level <- function(res_list, aliases = TRUE, blocked = TRUE) {

    for (i in seq_along(res_list)) {
      cat("      DESIGN RANK", i, "\n")
      cat("Generators:", paste(res_list[[i]]$generators_str, collapse=", "), "\n\n")
      cat("Fractional factorial Design:\n")
      print(res_list[[i]]$design, row.names = FALSE)
      cat("\nScores:\n"); print(res_list[[i]]$score)
      if (!is.null(res_list[[i]]$key)) cat("Key:", res_list[[i]]$key, "\n")
      if (aliases) {
        cat("\nAlias Structure:\n")
        a <- list_alias_structure_2level(res_list[[i]]$design)
        if (length(a) == 0) cat("  None\n") else for (j in seq_along(a)) cat("  ", paste(a[[j]], collapse=" = "), "\n")
      }
      if (blocked) {
        cat("\nBlocked Fractional Factorial Design:\n")
        blk <- make_blocked_design_2level(res_list[[i]]$design, auto = TRUE)
      }
      cat("------------------------------\n\n")
    }
  }

  cmp_key <- function(x, y) {
    for (i in seq_along(x)) {
      if (x[i] < y[i]) return(-1)
      if (x[i] > y[i]) return(1)
    }
    0
  }

  # ------------------------------------------------------------
  # Validate input & setup
  # ------------------------------------------------------------
  if (!is.numeric(n) || !is.numeric(k)) stop("n and k must be numeric")
  r <- n - k
  if (r <= 0) stop("n-k must be >= 1")
  if (k <= 0) stop("k must be >= 1")

  base <- gen_2level_from_base(r)
  base_names <- colnames(base)

  # Beam search nodes: list(gens, design, score, key)
  beam <- list(list(gens = list(), design = base, key = design_canonical_key_binary(base), score = NULL))

  depth <- 0
  while (depth < k) {
    depth <- depth + 1
    if (verbose) cat("Depth", depth, " | Beam size:", length(beam), "\n")
    next_beam <- list()

    for (node in beam) {
      p_now <- ncol(node$design)
      masks <- enumerate_masks_dynamic(p_now)
      if (length(masks) == 0) next

      # Canonical ordering: codes for new column
      codes <- sapply(masks, function(mask) {
        tmp <- add_dependent_col_from_mask(node$design, mask, "TMP")
        col_code_binary(tmp$TMP)
      })

      ord_masks <- order(codes)
      masks <- masks[ord_masks]
      codes <- codes[ord_masks]

      for (mask in masks) {
        new_design <- add_dependent_col_from_mask(node$design, mask, paste0("G", depth))
        new_key <- design_canonical_key_binary(new_design)
        scr <- compute_Aproxy_2level(new_design)
        next_beam[[length(next_beam)+1]] <- list(gens = c(node$gens, list(mask)), design = new_design, key = new_key, score = scr)
      }
    }

    if (length(next_beam) == 0) break

    scoremat <- do.call(rbind, lapply(next_beam, function(x) c(x$score["K3"], x$score["K4"])))
    ord <- order(scoremat[,1], scoremat[,2])

    # Keep only unique canonical keys
    beam <- list(); seen <- new.env(parent = emptyenv())
    for (idx in ord) {
      key <- next_beam[[idx]]$key
      if (exists(key, envir = seen, inherits = FALSE)) next
      assign(key, TRUE, envir = seen)
      beam[[length(beam)+1]] <- next_beam[[idx]]
      if (length(beam) >= beam_width) break
    }
  }

  # Final selection (unique)
  final_seen <- new.env(parent = emptyenv()); finals <- list()
  for (b in beam) {
    if (exists(b$key, envir = final_seen, inherits = FALSE)) next
    assign(b$key, TRUE, envir = final_seen)
    finals[[length(finals)+1]] <- b
  }
  if (length(finals) == 0) return(list())

  scoremat <- do.call(rbind, lapply(finals, function(x) c(x$score["K3"], x$score["K4"])))
  ord <- order(scoremat[,1], scoremat[,2])
  finals <- finals[ord[1:min(max_results, length(ord))]]

  # Convert mask lists to readable generators & rename columns
  output <- lapply(finals, function(x) {
    gens <- x$gens
    gens_str <- sapply(gens, function(m) {
      idx <- which(m == 1)
      paste(make_factor_names(ncol(base))[idx], collapse = "*")
    })
    design <- x$design
    colnames(design) <- c(make_factor_names(r), paste0("G", seq_len(ncol(design)-r)))
    list(generators_str = gens_str, design = design, score = x$score, key = x$key)
  })

  # Rename generated columns to letters for final output
  for (i in seq_along(output)) output[[i]]$design <- rename_generated_columns(output[[i]]$design, r)

  # Print clean output if requested
  if (verbose) print_clean_two_level(output)

  # Return results
  invisible(output)
}

