% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pid.R
\name{pid}
\alias{pid}
\alias{print.pid}
\alias{summary.pid}
\alias{print.summary.pid}
\title{Conformal PID control method}
\usage{
pid(
  object,
  alpha = 1 - 0.01 * object$level,
  symmetric = FALSE,
  ncal = 10,
  rolling = FALSE,
  integrate = TRUE,
  scorecast = !symmetric,
  scorecastfun = NULL,
  lr = 0.1,
  Tg = NULL,
  delta = NULL,
  Csat = 2/pi * (ceiling(log(Tg) * delta) - 1/log(Tg)),
  KI = max(abs(object$errors), na.rm = TRUE),
  ...
)
}
\arguments{
\item{object}{An object of class \code{"cvforecast"}. It must have an argument
\code{x} for original univariate time series, an argument \code{MEAN} for
point forecasts and \code{ERROR} for forecast errors on validation set.
See the results of a call to \code{\link{cvforecast}}.}

\item{alpha}{A numeric vector of significance levels to achieve a desired
coverage level \eqn{1-\alpha}.}

\item{symmetric}{If \code{TRUE}, symmetric nonconformity scores (i.e. \eqn{|e_{t+h|t}|})
are used. If \code{FALSE}, asymmetric nonconformity scores (i.e. \eqn{e_{t+h|t}})
are used, and then upper bounds and lower bounds are produced separately.}

\item{ncal}{Length of the burn-in period for training the scorecaster.
If \code{rolling = TRUE}, it is also used as the length of the trailing windows
for learning rate calculation and the windows for the calibration set.
If \code{rolling = FALSE}, it is used as initial period of calibration sets
and trailing windows for learning rate calculation.}

\item{rolling}{If \code{TRUE}, a rolling window strategy will be adopted to
form the trailing window for learning rate calculation and the calibration set
for scorecaster if applicable. Otherwise, expanding window strategy will be used.}

\item{integrate}{If \code{TRUE}, error integration will be included in the
update process.}

\item{scorecast}{If \code{TRUE}, scorecasting will be included in the update
process, and \code{scorecastfun} should be given.}

\item{scorecastfun}{A scorecaster function to return an object of class
\code{forecast}. Its first argument must be a univariate time series, and
it must have an argument \code{h} for the forecast horizon.}

\item{lr}{Initial learning rate used for quantile tracking.}

\item{Tg}{The time that is set to achieve the target absolute coverage
guarantee before this.}

\item{delta}{The target absolute coverage guarantee is set to \eqn{1-\alpha-\delta}.}

\item{Csat}{A positive constant ensuring that by time \code{Tg}, an absolute
guarantee is of at least \eqn{1-\alpha-\delta} coverage.}

\item{KI}{A positive constant to place the integrator on the same scale as the scores.}

\item{...}{Other arguments are passed to the \code{scorecastfun} function.}
}
\value{
A list of class \code{c("pid", "cpforecast", "forecast")}
with the following components:
\item{x}{The original time series.}
\item{series}{The name of the series \code{x}.}
\item{method}{A character string "pid".}
\item{cp_times}{The number of times the conformal prediction is performed in
cross-validation.}
\item{MEAN}{Point forecasts as a multivariate time series, where the \eqn{h}th column
holds the point forecasts for forecast horizon \eqn{h}. The time index
corresponds to the period for which the forecast is produced.}
\item{ERROR}{Forecast errors given by
\eqn{e_{t+h|t} = y_{t+h}-\hat{y}_{t+h|t}}{e[t+h] = y[t+h]-f[t+h]}.}
\item{LOWER}{A list containing lower bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{UPPER}{A list containing upper bounds for prediction intervals for
each \code{level}. Each element within the list will be a multivariate time
series with the same dimensional characteristics as \code{MEAN}.}
\item{level}{The confidence values associated with the prediction intervals.}
\item{call}{The matched call.}
\item{model}{A list containing information abouth the conformal prediction model.}
If \code{mean} is included in the \code{object}, the components \code{mean},
\code{lower}, and \code{upper} will also be returned, showing the information
about the forecasts generated using all available observations.
}
\description{
Compute prediction intervals and other information by
applying the conformal PID (Proportional-Integral-Derivative) control method.
}
\details{
The PID method combines three modules to make the final iteration:
\deqn{q_{t+h|t}=\underbrace{q_{t+h-1|t-1} + \eta(\mathrm{err}_{t|t-h}-\alpha)}_{\mathrm{P}}+\underbrace{r_t\left(\sum_{i=1}^t\left(\mathrm{err}_{i|i-h}-\alpha\right)\right)}_{\mathrm{I}}+\underbrace{\hat{s}_{t+h|t}}_{\mathrm{D}}}
for each individual forecast horizon \code{h}, respectively, where
\itemize{
\item Quantile tracking part (P) is \eqn{q_{t+h-1|t-1} + \eta(\mathrm{err}_{t|t-h}-\alpha)}, where \eqn{q_{1+h|1}} is set to 0 without a loss of generality, \eqn{\mathrm{err}_{t|t-h}=1} if \eqn{s_{t|t-h}>q_{t|t-h}}, and \eqn{\mathrm{err}_{t|t-h}=0} if \eqn{s_{t|t-h} \leq q_{t|t-h}}.
\item Error integration part (I) is \eqn{r_t\left(\sum_{i=1}^t\left(\mathrm{err}_{i|i-h}-\alpha\right)\right)}. Here we use a nonlinear saturation
function \eqn{r_t(x)=K_{\mathrm{I}} \tan \left(x \log (t) /\left(t C_{\text {sat }}\right)\right)}, where we set \eqn{\tan (x)=\operatorname{sign}(x) \cdot \infty} for \eqn{x \notin[-\pi / 2, \pi / 2]}, and \eqn{C_{\text {sat }}, K_{\mathrm{I}}>0} are constants that we choose heuristically.
\item Scorecasting part (D) is \eqn{\hat{s}_{t+h|t}} is forecast generated
by training a scorecaster based on nonconformity scores available at time \eqn{t}.
}
}
\examples{
# Simulate time series from an AR(2) model
library(forecast)
series <- arima.sim(n = 200, list(ar = c(0.8, -0.5)), sd = sqrt(1))
# Cross-validation forecasting
far2 <- function(x, h, level) {
  Arima(x, order = c(2, 0, 0)) |>
    forecast(h = h, level)
}
fc <- cvforecast(series, forecastfun = far2, h = 3, level = 95,
                 forward = TRUE, initial = 1, window = 50)
# PID setup
Tg <- 200; delta <- 0.01
Csat <- 2 / pi * (ceiling(log(Tg) * delta) - 1 / log(Tg))
KI <- 2
lr <- 0.1
# PID without scorecaster
pidfc_nsf <- pid(fc, symmetric = FALSE, ncal = 50, rolling = TRUE,
                 integrate = TRUE, scorecast = FALSE,
                 lr = lr, KI = KI, Csat = Csat)
print(pidfc_nsf)
summary(pidfc_nsf)
# PID with a Naive model for the scorecaster
naivefun <- function(x, h) {
  naive(x) |> forecast(h = h)
}
pidfc <- pid(fc, symmetric = FALSE, ncal = 50, rolling = TRUE,
             integrate = TRUE, scorecast = TRUE, scorecastfun = naivefun,
             lr = lr, KI = KI, Csat = Csat)

}
\references{
Angelopoulos, A., Candes, E., and Tibshirani, R. J. (2024).
"Conformal PID control for time series prediction", \emph{Advances in Neural
Information Processing Systems}, \bold{36}, 23047--23074.
}
