% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/core_estimate.R, R/formatting_fits_agg_dm.R,
%   R/formatting_fits_ids_dm.R, R/formatting_mcmc_dm.R
\name{estimate_dm}
\alias{estimate_dm}
\alias{print.fits_agg_dm}
\alias{print.fits_ids_dm}
\alias{print.mcmc_dm}
\title{Fit a DDM to Observed Data}
\usage{
estimate_dm(
  drift_dm_obj,
  obs_data = NULL,
  approach = NULL,
  optimizer = NULL,
  control = list(),
  n_cores = 1,
  parallelization_strategy = NULL,
  lower = NULL,
  upper = NULL,
  start_vals = NULL,
  means = NULL,
  sds = NULL,
  shapes = NULL,
  rates = NULL,
  n_chains = 40,
  burn_in = 500,
  samples = 1000,
  prob_migration = 0.1,
  prob_re_eval = 1,
  messaging = TRUE,
  seed = NULL,
  ...
)

\method{print}{fits_agg_dm}(x, ...)

\method{print}{fits_ids_dm}(x, ...)

\method{print}{mcmc_dm}(x, ..., round_digits = drift_dm_default_rounding())
}
\arguments{
\item{drift_dm_obj}{a \link{drift_dm} object containing the model to be
fitted.}

\item{obs_data}{an optional \link{data.frame} (see also \link{obs_data}).
If no \code{ID} column is present, a single-individual setup is assumed.
If an \code{ID} column is present, the model is fitted separately for each
individual.}

\item{approach}{an optional character string, specifying the approach to
fitting the model. Options are \code{"sep_c"}, \code{"agg_c"}, \code{"sep_b"}, \code{"hier_b"}
(see the Details).}

\item{optimizer}{a character string. For classical optimization, one of
\code{"nmkb"}, \code{"Nelder-Mead"}, \code{"BFGS"}, \code{"L-BFGS-B"}, \code{"DEoptim"}. For the
Bayesian framework, only \code{"DE-MCMC"} is currently supported. If \code{NULL} and
if a classical optimization approach is used, defaults to \code{"DEoptim"} or
\code{"Nelder-Mead"}, depending on whether \code{lower/upper} are provided or not. If
\code{NULL} and if a Bayesian framework is used, defaults to \verb{"DE-MCMC}.
Note that \code{"BFGS"} and \code{"L-BFGS-B"} are often unstable.}

\item{control}{a list of control parameters passed to the optimizer
(for Nelder-Mead, BFGS, and L-BFGS-B, see \link[stats:optim]{stats::optim}; for nmkb, see
\link[dfoptim:nmkb]{dfoptim::nmkb}; for DEoptim, see \link[DEoptim:DEoptim]{DEoptim::DEoptim}).
Per default, we set the \code{trace} control argument for \link[DEoptim:DEoptim]{DEoptim::DEoptim} to
\code{FALSE}. Also, we set the \code{parscale} control argument for "Nelder-Mead" via
\link[stats:optim]{stats::optim} to \code{pmax(x0, 1e-6)}.}

\item{n_cores}{an integer > 0, indicating the number of CPU cores/threads to
use (at the moment, this doesn't have an effect when fitting a single
individual within the Bayesian framework).}

\item{parallelization_strategy}{an integer, controlling how parallelization
is performed when fitting multiple individuals with the classical approach.
If \code{1}, parallelization is across individuals. If \code{2}, parallelization is
within individuals (currently only supported for \code{"DEoptim"}). Defaults to
\code{1}.}

\item{lower, upper}{numeric vectors or lists, specifying the lower and upper
bounds on each parameter to be optimized (see Details).}

\item{start_vals}{optional starting values for classical single-subject fits
and when using an optimizer that requires a starting value. Can be
a numeric vector of model parameters when fitting a single individual, or
a \code{data.frame} with columns for each model parameter. In the latter case,
enables multi-start (one row per start). For \code{'approach = "separately"'}, a
\code{data.frame} with an \code{ID} column is required.}

\item{means, sds, shapes, rates}{optional numeric vectors for prior
specification (when using the Bayesian framework, see Details).}

\item{n_chains}{an integer, providing the number of MCMC chains (Bayesian
framework).}

\item{burn_in}{an integer, number of burn-in iterations (Bayesian
framework).}

\item{samples}{an integer, number of post-burn-in samples per chain (
Bayesian framework).}

\item{prob_migration}{a numeric in \verb{[0,1]}, controlling the migration
probability of the \code{DE-MCMC} algorithm (Bayesian framework).}

\item{prob_re_eval}{a numeric in \verb{[0,1]}, probability to re-evaluate the
model at current group-level parameters during sampling (Bayesian
framework; only relevant for the hierarchical case).}

\item{messaging}{a logical, if \code{TRUE} progress/info messages are printed}

\item{seed}{an optional integer to set the RNG seed for reproducibility.}

\item{...}{additional arguments forwarded to lower-level routines. Options
are: \code{progress/verbose} (integers, for controlling progress bars and
verbosity of estimation infos), \code{round_digits} (for controlling the number
of digits for rounding when printing individual model evaluations;
if \code{verbose = 2}), \code{return_runs}
(when fitting a single individual and starting the estimation
routine with multiple starting points; if \code{TRUE}, then a list of all
routines is returned), \code{probs/n_bins} (the quantile levels and the number
of CAF bins when fitting aggregated data using the RMSE cost function),
\code{use_ez/n_lhs} (logical and integer; the first controls if EZ-Diffusion
Parameter Estimates shall be used for determining starting points; the
latter controls the number of parameters to sample per dimension for the
latin hypercube sampling when searching for starting values)}

\item{x}{an object of type \code{fits_agg_dm}, \code{fits_ids_dm}, or \code{mcmc_dm}}

\item{round_digits}{integer, specifying the number of decimal places for
rounding in the printed summary. Default is 3.}
}
\value{
\itemize{
\item If fitting a single individual: either a \code{drift_dm} object with
fitted parameters and additional fit information (for the classical
optimization framework) or an object of type \code{mcmc_dm} (for the Bayesian
framework)
\item If fitting multiple individuals separately: a \code{fits_ids_dm} object
or a list of \code{mcmc_dm} objects, containing all the individual model fits.
\item If fitting aggregated data: a \code{fits_agg_dm} object containing the model
itself and the raw data.
\item If fitting multiple individuals hierarchically: an object of type
\code{mcmc_dm}.
}
}
\description{
\code{estimate_dm()} is the main function to fit a drift diffusion model (DDM)
in \code{dRiftDM}. Several ways of fitting a model are supported: fitting a single
participant, fitting multiple participants separately or aggregated, and
fitting a (hierarchical) Bayesian model. The particular way is controlled
via the \code{approach} argument.
}
\details{
\subsection{Fitting Approaches}{

The function supports different "approaches" to fitting data.
\itemize{
\item \code{"sep_c"}: This means that data is always considered \code{sep}arately for
each participant (if there are multiple participants) and that a
\code{c}lassical approach to parameter optimization is used. This means that
a standard \link{cost_function} is minimized (e.g., the negative
log-likelihood). If users provide only a single participant or a data set
without an \code{ID} column, then the model is fitted just once to that data
set.
\item \code{"agg_c"}: This fits the model to aggregated data. For each individual in
a data set, summary statistics (e.g., quantiles, accuracies) are
calculated, and the model is fitted once to the average of these summary
statistics.
\item \code{"sep_b"}: Similar to \verb{sep_b"}, although a Bayesian approach is used to
sample from the posterior distribution.
\item \code{"hier_b"}: A hierarchical approach to parameter estimation. In this case
all participants are considered simultaneously and samples are drawn both
at the individual-level and group-level.
}

The optimizers  \code{"nmkb"}, \code{"L-BFGS-B"}, and \code{"DEoptim"} (for classical
parameter optimization) require the specification of the \code{lower/upper}
arguments.
}

\subsection{Fitting to Aggregated Data}{

For aggregated fits, aggregated statistics are set to the model and the cost
function is switched to \code{"rmse"}. If incompatible settings are requested,
the function switches to a compatible configuration and informs the user
with messages (these messages can be suppressed via the \code{messaging} argument).
}

\subsection{Specifying \code{lower/upper} for Classical optimization}{

the function \code{estimate_model_dm()} provides a flexible way of specifying the
optimization space; this is identical to specifying the parameter simulation
space in \code{\link[=simulate_data.drift_dm]{simulate_data.drift_dm()}}.

Users have three options to specify the search space (see also the examples
below):
\itemize{
\item Plain numeric vectors (not very much recommended). In this case,
\code{lower/upper} must be sorted in accordance with the parameters in the
underlying \link{flex_prms} object of \code{drift_dm_obj} that vary for at
least one condition (call \code{print(drift_dm_obj)} and have a look at the
columns of the \verb{Parameter Settings} output; for each column that has a
number > 0, specify an entry in \code{lower/upper}).
\item Named numeric vectors. In this case \code{lower/upper} have to provide labels
in accordance with the parameters that are considered "free" at least once
across conditions (call \code{coef(drift_dm_obj)} and provide one named entry for
each parameter; dRiftDM will try to recycle parameter values across
conditions).
\item The most precise way is when \code{lower/upper} are lists. In this case, the
list requires an entry called "default_values" which specifies the named or
plain numeric vectors as above. If the list only contains this entry, then
the behavior is as if \code{lower/upper} were already numeric vectors. However,
the \code{lower/upper} lists can also provide entries labeled as specific
conditions, which contain named (!) numeric vectors with parameter labels.
This will modify the value for the upper/lower parameter space with respect
to the specified parameters in the respective condition.
}
}

\subsection{Specifying Priors for Bayesian Estimation}{

\strong{(Default) Prior settings in the non-hierarchical case:}

Let \eqn{\theta^{(j)}} indicate parameter \eqn{j} of a model (e.g., the
drift rate).
The prior on \eqn{\theta^{(j)}} is a truncated normal distribution:
\deqn{
 \theta^{(j)} \sim NT(\mu^{(j)}, \sigma^{(j)}, l^{(j)}, u^{(j)})
 }
With \eqn{\mu^{(j)}} and \eqn{\sigma^{(j)}} representing the mean and standard
deviation of parameter \eqn{j}. \eqn{l^{(j)}} and \eqn{u^{(j)}} represent the
lower and upper boundary. \eqn{\mu^{(j)}} is taken from the \code{mean}
argument or the currently set model parameters (i.e., from
\code{coef(drift_dm_obj)}) when calling the function. \eqn{\sigma^{(j)}} is, per
default, equal to \eqn{\mu^{(j)}}. This can be changed by passing
the \code{sd} argument. The lower and upper boundaries of the truncated normal
are \code{-Inf} and \code{Inf} per default. This can be altered by passing the
arguments \code{lower} and  \code{upper} (see the examples below).

\strong{(Default) Prior settings in the hierarchical case:}

Let \eqn{\theta_i^{(j)}} indicate parameter \eqn{j} for participant \eqn{i}
(e.g., the  drift rate estimated for individual \eqn{i}). The prior on
\eqn{\theta_i^{(j)}} is a truncated normal distribution:
\deqn{
 \theta_i^{(j)} \sim NT(\mu^{(j)}, \sigma^{(j)}, l^{(j)}, u^{(j)})
 }
With \eqn{\mu^{(j)}} and \eqn{\sigma^{(j)}} representing the mean and
standard deviation of parameter \eqn{j} at the group level. \eqn{l^{(j)}} and
\eqn{u^{(j)}} represent the lower and upper boundary. The lower and upper
boundaries of the truncated normal are \code{-Inf} and \code{Inf} per default.
This can be altered by passing the arguments \code{lower} and  \code{upper}.

For a group-level mean parameter, \eqn{\mu^{(j)}}, the prior is also a
truncated normal distributions:
\deqn{
 \mu^{(j)} \sim NT(M^{(j)}, SD^{(j)}, l^{(j)}, u^{(j)})
 }
With \eqn{M^{(j)}} specified by the \code{mean} argument or the currently
set model parameters. \eqn{SD^{(j)}} is, per default, equal to \eqn{M^{(j)}}.
This can be changed by passing the \code{sd} argument.

For a group-level standard deviation parameter, \eqn{\sigma^{(j)}}, the prior
is a gamma distribution:
\deqn{
 \sigma^{(j)} \sim \Gamma(shape^{(j)},rate^{(j)})
 }
With \eqn{shape^{(j)}} and \eqn{rate^{(j)}} being \code{1} by default. This
can be changed by passing the arguments \code{shape} and \code{rate}.

\strong{Specifying Prior Settings/Arguments}

Argument specification for \code{mean}, \code{sd}, \code{lower}, \code{upper}, \code{shape} and
\code{rate} is conceptually identical to specifying \code{lower/upper} for the
classical optimization approach (see the subsection above and the examples
below).
}
}
\note{
\code{estimate_dm} dispatches to underlying estimation routines that are not
exported:
\itemize{
\item Classical optimization of one individual via
\code{\link[=estimate_classical]{estimate_classical()}}
\item Classical optimization of multiple individuals via
\code{\link[=estimate_classical_wrapper]{estimate_classical_wrapper()}}
\item Bayesian estimation via \code{\link[=estimate_bayesian]{estimate_bayesian()}}.
\item Aggregated fitting is handled within \code{estimate_dm()} in combination with
\code{\link[=estimate_classical]{estimate_classical()}}
}

When fitting a model with \code{optimizer = "DEoptim"}, the corresponding
minimization routine always runs for 200 iterations by default, irrespective
of whether a minimum has already been reached (see
\link[DEoptim:DEoptim.control]{DEoptim::DEoptim.control}). Therefore, with default optimization settings,
\code{estimate_dm()} returns the convergence flag \code{NA} for
\code{optimizer = "DEoptim"}, because the termination of the routine does not
necessarily indicate convergence. However, this is typically not an issue, as
200 iterations are generally sufficient for the algorithm to find the global
minimum. If users explicitly define convergence criteria via the \code{control}
argument of \code{estimate_dm()} (which is passed on to
\link[DEoptim:DEoptim.control]{DEoptim::DEoptim.control}), valid convergence messages and flags are
returned.
}
\examples{
##########
# Note: The following examples were trimmed for speed to ensure they run
# within seconds. They do not always provide realistic settings.
##########

####
# Setup

# get a model for the examples (DMC with just two free parameters)
model <- dmc_dm(
  instr = "
   b <!>
   non_dec <!>
   sd_non_dec <!>
   tau <!>
   alpha <!>
   "
)

# get some data (the first two participants in the data set of Ulrich et al.)
data <- ulrich_flanker_data[ulrich_flanker_data$ID \%in\% 1:2, ]


####
# Fit a single individual (using unbounded Nelder-Mead)
fit <- estimate_dm(
  drift_dm_obj = model,
  obs_data = data[data$ID == 1, ],
  optimizer = "Nelder-Mead"
)
print(fit)




####
# Fit a single individual (using DEoptim)
l_u <- get_lower_upper(model)
set.seed(2)
fit <- estimate_dm(
  drift_dm_obj = model,
  obs_data = data[data$ID == 1, ],
  optimizer = "DEoptim",
  lower = l_u$lower, upper = l_u$upper,
  control = list(itermax = 5) # use default itermax in practice!
)
print(fit)


####
# Fit multiple individuals (separately; using bounded Nelder-Mead)
l_u <- get_lower_upper(model)
fit <- estimate_dm(
  drift_dm_obj = model,
  obs_data = data, # contains the data for two individuals
  optimizer = "nmkb",
  lower = l_u$lower, upper = l_u$upper,
)
print(fit)
coef(fit)


###
# Fit to aggregated data (using unbounded Nelder-Mead)
fit <- estimate_dm(
  drift_dm_obj = model,
  obs_data = data, # contains data for two individuals
  optimizer = "Nelder-Mead",
  approach = "agg_c"
)
print(fit)
coef(fit)


###
# EXPERIMENTAL
# Fit a single individual (using DE-MCMC; Bayesian; custom priors)
fit <- estimate_dm(
  drift_dm_obj = model,
  obs_data = data[data$ID == 1, ],
  approach = "sep_b",
  burn_in = 1, # higher in practice (e.g., 500)
  samples = 1, # higher in practice (e.g., 1000)
  n_chains = 5, # higher in practice (e.g., 40)
  mean = c(muc = 3, A = 0.9),
  sd = c(muc = 2, A = 0.8),
)
print(fit)
coef(fit)


###
# EXPERIMENTAL
# Fit multiple individuals (using DE-MCMC; hierarchical Bayesian)
fit <- estimate_dm(
  drift_dm_obj = model,
  approach = "hier_b",
  obs_data = data, # contains data for two individuals
  burn_in = 1, # higher in practice (e.g., 500)
  samples = 1, # higher in practice (e.g., 1000)
  n_chains = 5, # higher in practice (e.g., 40)
  n_cores = 1, # higher in practice (depending on your machine and data set)
)
print(fit)
coef(fit)

}
\seealso{
\code{\link[=estimate_classical]{estimate_classical()}}, \code{\link[=estimate_bayesian]{estimate_bayesian()}},
\code{\link[=estimate_classical_wrapper]{estimate_classical_wrapper()}}, \code{\link[=get_parameters_smart]{get_parameters_smart()}}
}
