% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/power.r
\name{power}
\alias{power}
\title{Simulated power}
\usage{
power(data, ..., alpha = 0.05, list_column = FALSE, ncores = 1L)
}
\arguments{
\item{data}{(depower)\cr
The simulated data returned by \code{\link[=sim_log_lognormal]{sim_log_lognormal()}}, \code{\link[=sim_nb]{sim_nb()}}, or \code{\link[=sim_bnb]{sim_bnb()}}.
In each, argument \code{return_type} must be the default \code{"list"}.}

\item{...}{(functions)\cr
The function(s) used to perform the test. If empty, a default test
function will be selected based on \code{class(data)}. Names are used for
labeling and should be unique. If names are empty, the call coerced to
character will be used for name-value pairs. See 'Details'.}

\item{alpha}{(numeric: \code{0.05}; \verb{(0,1)})\cr
The expected probability of rejecting a single null hypothesis when it
is actually true. See 'Details'.}

\item{list_column}{(Scalar logical: \code{FALSE})\cr
If \code{TRUE}, the \code{data} and \code{result} list-columns are included in the
returned data frame. If \code{FALSE} (default), the \code{data} and \code{result}
list-columns are not included in the returned data frame.}

\item{ncores}{(Scalar integer: \code{1L}; \verb{[1,Inf)})\cr
The number of cores (number of worker processes) to use. Do not set
greater than the value returned by \code{\link[parallel:detectCores]{parallel::detectCores()}}. May be
helpful when the number of parameter combinations is large and \code{nsims}
is large.}
}
\value{
A data frame with the following columns appended to the \code{data} object:
\tabular{ll}{
Name \tab Description \cr
\code{alpha}  \tab Type I assertion probability. \cr
\code{test}   \tab Name-value pair of the function and statistical test: \verb{c(as.character(...) = names(...)}. \cr
\code{data}   \tab List-column of simulated data. \cr
\code{result} \tab List-column of test results. \cr
\code{power}  \tab Power of the test for corresponding parameters.
}

For \code{power(list_column = FALSE)}, columns \code{data}, and \code{result} are excluded from
the data frame.
}
\description{
A method to calculate power for objects returned by \code{\link[=sim_log_lognormal]{sim_log_lognormal()}},
\code{\link[=sim_nb]{sim_nb()}}, and \code{\link[=sim_bnb]{sim_bnb()}}.
}
\details{
Power is calculated as the proportion of hypothesis tests which result in a
p-value less than or equal to \code{alpha}. e.g.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sum(p <= alpha) / nsims
}\if{html}{\out{</div>}}

Power is defined as the expected probability of rejecting the null hypothesis
for a chosen value of the unknown effect. In a multiple comparisons scenario,
power is defined as the marginal power, which is the expected power of the
test for each individual null hypothesis assumed to be false.

Other forms of power under the multiple comparisons scenario include
disjunctive or conjunctive power. Disjunctive power is defined as the
expected probability of correctly rejecting one or more null hypotheses.
Conjunctive power is defined as the expected probability of correctly
rejecting all null hypotheses. In the simplest case, and where all hypotheses
are independent, if the marginal power is defined as \eqn{\pi} and \eqn{m} is
the number of null hypotheses assumed to be false, then disjunctive power may
be calculated as \eqn{1 - (1 - \pi)^m} and conjunctive power may be calculated
as \eqn{\pi^m}. Disjunctive power tends to decrease with increasingly
correlated hypotheses and conjunctive power tends to increase with
increasingly correlated hypotheses.
\subsection{Argument \code{...}}{

\code{...} are the name-value pairs for the functions used to perform the tests.
If not named, the functions coerced to character will be used for the
name-value pairs. Typical in non-standard evaluation, \code{...} accepts bare
functions and converts them to a list of expressions. Each element in this
list will be validated as a \code{call} and then evaluated on the simulated data.
A \code{\link[base:call]{base::call()}} is simply an unevaluated function. Below are some examples
of specifying \code{...} in \code{power()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Examples of specifying ... in power()
data <- sim_nb(
  n1 = 10,
  mean1 = 10,
  ratio = c(1.6, 2),
  dispersion1 = 2,
  dispersion2 = 2,
  nsims = 200
)

# ... is empty, so an appropriate default function will be provided
power(data)

# This is equivalent to leaving ... empty
power(data, "NB Wald test" = wald_test_nb())

# If not named, "wald_test_nb()" will be used to label the function
power(data, wald_test_nb())

# You can specify any parameters in the call. The data argument
# will automatically be inserted or overwritten.
data |>
  power("NB Wald test" = wald_test_nb(equal_dispersion=TRUE, link="log"))

# Multiple functions may be used.
data |>
  power(
    wald_test_nb(link='log'),
    wald_test_nb(link='sqrt'),
    wald_test_nb(link='squared'),
    wald_test_nb(link='identity')
  )

# Just like functions in a pipe, the parentheses are required.
# This will error because wald_test_nb is missing parentheses.
try(power(data, wald_test_nb))
}\if{html}{\out{</div>}}

In most cases*, any user created test function may be utilized in \code{...} if the
following conditions are satisfied:
\enumerate{
\item The function contains argument \code{data} which is defined as a list with the
first and second elements for simulated data.
\item The return object is a list with element \code{p} for the p-value of the
hypothesis test.
}

Validate with test cases beforehand.

*Simulated data of class \code{log_lognormal_mixed_two_sample} has both independent
and dependent data. To ensure the appropriate test function is used,
\code{power.log_lognormal_mixed_two_sample()} allows only
\code{\link[=t_test_welch]{t_test_welch()}} and \code{\link[=t_test_paired]{t_test_paired()}} in \code{...}. Each will
be evaluated on the simulated data according to column \code{data$cor}. If one or
both of these functions are not included in \code{...}, the corresponding default
function will be used automatically. If any other test function is included,
an error will be returned.
}

\subsection{Argument \code{alpha}}{

\eqn{\alpha} is known as the type I assertion probability and is defined as
the expected probability of rejecting a null hypothesis when it was actually
true. \eqn{\alpha} is compared with the p-value and used as the decision
boundary for rejecting or not rejecting the null hypothesis.

The family-wise error rate is the expected probability of making one or more
type I assertions among a family of hypotheses. Using Bonferroni's method,
\eqn{\alpha} is chosen for the family of hypotheses then divided by the
number of tests performed (\eqn{m}). Each individual hypothesis is tested at
\eqn{\frac{\alpha}{m}}. For example, if you plan to conduct 30 hypothesis
tests and want to control the family-wise error rate to no greater than
\eqn{\alpha = 0.05}, you would set \code{alpha = 0.05/30}.

The per-family error rate is the expected number of type I assertions among a
family of hypotheses. If you calculate power for the scenario where you
perform 1,000 hypotheses and want to control the per-family error rate to no
greater than 10 type I assertions, you would choose \code{alpha = 10/1000}. This
implicitly assumes all 1,000 hypotheses are truly null. Alternatively, if you
assume 800 of these hypotheses are truly null and 200 are not,
\code{alpha = 10/1000} would control the per-family error rate to no greater than
8 type I assertions. If it is acceptable to keep the per-family error rate as
10, setting \code{alpha = 10/800} would provide greater marginal power than the
previous scenario.

These two methods assume that the distribution of p-values for the truly null
hypotheses are uniform(0,1), but remain valid under various other testing
scenarios (such as dependent tests). Other multiple comparison methods, such
as FDR control, are common in practice but don't directly fit into this
power simulation framework.
}

\subsection{Column \code{nsims}}{

The final number of valid simulations per unique set of simulation parameters
may be less than the original number requested. This may occur when the test
results in a missing p-value. For \code{wald_test_bnb()}, pathological MLE
estimates, generally from small sample sizes and very small dispersions, may
result in a negative estimated standard deviation of the ratio. Thus the test
statistic and p-value would not be calculated. Note that simulated data from
\code{sim_nb()} and \code{sim_bnb()} may also reduce \code{nsims} during the data simulation
phase.

\code{nsims} denotes the effective number of simulated datasets under the alternative hypothesis, resulting in the equivalent number of hypothesis tests performed used to calculate power.
If \code{nsims} is too small, the power estimate will have high uncertainty (wide confidence/prediction intervals).
If \code{nsims} is too large, computation time may be prohibitive.
To aid in choosing an appropriate \code{nsims}, functions \code{\link[=eval_power_ci]{eval_power_ci()}} and \code{\link[=eval_power_pi]{eval_power_pi()}} are helpful to understand the precision of the interval for power, before simulation takes place.
Their counterparts, \code{\link[=add_power_ci]{add_power_ci()}} and \code{\link[=add_power_pi]{add_power_pi()}} add intervals for power to the object returned by \code{\link[=power]{power()}}.
Functions \code{\link[=eval_power_ci]{eval_power_ci()}} and \code{\link[=add_power_ci]{add_power_ci()}} quantify uncertainty in the true power parameter, and answer the question, "What is the plausible range of true power values given my simulation results?"
Functions \code{\link[=eval_power_pi]{eval_power_pi()}} and \code{\link[=add_power_pi]{add_power_pi()}} quantify the expected range of power estimates from a future simulation study, and answer the question, "If I run a new simulation study with \eqn{m} simulations, what range of power estimates might I observe?" which is particularly useful for deciding the optimal \code{nsims}.
When the prediction intervals from \code{\link[=eval_power_pi]{eval_power_pi()}} are too wide, consider choosing a larger \code{nsims} before running a power simulation.
}
}
\examples{
#----------------------------------------------------------------------------
# power() examples
#----------------------------------------------------------------------------
library(depower)

# Power for independent two-sample t-test
set.seed(1234)
data <- sim_log_lognormal(
  n1 = 20,
  n2 = 20,
  ratio = c(1.2, 1.4),
  cv1 = 0.4,
  cv2 = 0.4,
  cor = 0,
  nsims = 1000
)

# Welch's t-test is used by default
power(data)

# But you can specify anything else that is needed
power(
  data = data,
  "Welch's t-Test" = t_test_welch(alternative = "greater"),
  alpha = 0.01
)

# The 95\% posterior predictive interval for power based on 1000 simulations
power(data) |>
  add_power_pi()

# Power for dependent two-sample t-test
set.seed(1234)
sim_log_lognormal(
  n1 = 20,
  n2 = 20,
  ratio = c(1.2, 1.4),
  cv1 = 0.4,
  cv2 = 0.4,
  cor = 0.5,
  nsims = 1000
) |>
  power()

# Power for mixed-type two-sample t-test
set.seed(1234)
sim_log_lognormal(
  n1 = 20,
  n2 = 20,
  ratio = c(1.2, 1.4),
  cv1 = 0.4,
  cv2 = 0.4,
  cor = c(0, 0.5),
  nsims = 1000
) |>
  power()

# Power for one-sample t-test
set.seed(1234)
sim_log_lognormal(
  n1 = 20,
  ratio = c(1.2, 1.4),
  cv1 = 0.4,
  nsims = 1000
) |>
  power()

# Power for independent two-sample NB test
set.seed(1234)
sim_nb(
  n1 = 10,
  mean1 = 10,
  ratio = c(1.6, 2),
  dispersion1 = 2,
  dispersion2 = 2,
  nsims = 200
) |>
  power()

# Power for BNB test
set.seed(1234)
sim_bnb(
  n = 10,
  mean1 = 10,
  ratio = c(1.4, 1.6),
  dispersion = 10,
  nsims = 200
) |>
  power()

}
\references{
\insertRef{yu_2017}{depower}

\insertRef{yu_2020}{depower}

\insertRef{rettiganti_2012}{depower}

\insertRef{aban_2009}{depower}

\insertRef{julious_2004}{depower}

\insertRef{vickerstaff_2019}{depower}
}
\seealso{
\code{\link[=plot.depower]{plot.depower()}},
\code{\link[=add_power_ci]{add_power_ci()}},
\code{\link[=add_power_pi]{add_power_pi()}},
\code{\link[=eval_power_ci]{eval_power_ci()}},
\code{\link[=eval_power_pi]{eval_power_pi()}}
}
