#' Random generation from the exponential Factor Copula Model (eFCM)
#'
#' Draws \eqn{n} samples from the eFCM.
#'
#' @param lambda,delta Positive scalars: rate \eqn{\lambda} and range \eqn{\delta}.
#' @param dist Optional \eqn{d\times d} distance matrix. If \code{NULL}, provide \code{coord}.
#' @param coord Optional two-column matrix/data.frame of station coordinates (lon, lat).
#'   Used to build \code{dist} via \code{fields::rdist.earth()}.
#' @param nu Matérn smoothness parameter (default \code{0.5}).
#' @param n Number of simulated rows (default \code{5e5}).
#' @param miles Logical passed to \code{fields::rdist.earth} (default \code{FALSE}).
#' @param seed Optional integer seed for reproducibility.
#'
#' @return A numeric matrix of size \code{n x d} (rows = samples, cols = stations).
#' @examples
#' \donttest{
#' data(LonLat)
#' sim <- rmfcm(lambda = 2, delta = 100, coord = LonLat[1:2, ], n = 10000)
#' dim(sim)  # 10000 x 2
#' }
#' @importFrom fields rdist.earth
#' @export
rmfcm <- function(lambda, delta, dist = NULL, coord = NULL,
                  nu = 0.5, n = 5e5, miles = FALSE, seed = NULL) {
  stopifnot(is.numeric(lambda), length(lambda) == 1, lambda > 0,
            is.numeric(delta),  length(delta)  == 1, delta  > 0,
            is.numeric(nu),     length(nu)     == 1, nu     > 0,
            is.numeric(n),      length(n)      == 1, n > 0)

  if (is.null(dist)) {
    if (is.null(coord))
      stop("Provide either `dist` or `coord`.", call. = FALSE)
    if (!requireNamespace("fields", quietly = TRUE))
      stop("Package 'fields' is required when `coord` is supplied.", call. = FALSE)
    dist <- fields::rdist.earth(as.matrix(coord), miles = miles)
  } else {
    dist <- as.matrix(dist)
    if (nrow(dist) != ncol(dist)) stop("`dist` must be a square matrix.", call. = FALSE)
  }

  if (!is.null(seed)) set.seed(as.integer(seed))
  rmfcm_rcpp(lambda = lambda, delta = delta, dist = dist,
             nu = nu, n = as.integer(n))
}
