#' CSEM and CSSEM with Binomial Model
#'
#' @description
#' Compute the conditional standard error of measurement (CSEM) and conditional
#' standard error of scaled scores (CSSEM) under the binomial model.
#'
#' @param ni A single numeric value indicating the number of items.
#' @param ct An optional data frame or matrix containing a conversion table with
#'   two columns: the first column as raw scores (0 to \code{ni}) and the second
#'   column as scale scores.
#'
#' @details
#' Under the binomial model, for a test with \eqn{n_i} items and a true-score
#' proportion \eqn{\pi}, the distribution of raw scores is assumed to be
#' \eqn{\mathrm{Binomial}(n_i, \pi)}. This function treats each possible raw
#' score \eqn{k = 0, 1, \ldots, n_i} as the true-score value (i.e.,
#' \eqn{\pi_k = k / n_i}) and computes:
#' \itemize{
#'   \item the CSEM of the raw scores; and
#'   \item if \code{ct} is provided, the CSSEM of the scale scores defined in
#'     the conversion table.
#' }
#'
#' @return A list with:
#' \describe{
#'   \item{x}{A vector of raw scores from 0 to \code{ni}.}
#'   \item{csem}{A vector of CSEM values (on the raw-score metric) for each
#'     raw score.}
#'   \item{cssem}{If \code{ct} is provided, a vector of CSSEM values for the
#'     scale scores corresponding to each raw score.}
#' }
#'
#'
#' @examples
#' csem_binomial(40)
#' csem_binomial(40, ct.u)
#'
#' @export
csem_binomial <- function(ni, ct = NULL) {
  # basic checks ---------------------------------------------------------------
  if (missing(ni)) {
    stop("`ni` must be supplied as the number of items.")
  }
  if (!is.numeric(ni) || length(ni) != 1L || is.na(ni)) {
    stop("`ni` must be a single numeric value.")
  }

  ni <- as.integer(ni)
  if (ni < 2L) {
    stop("`ni` must be at least 2 to compute CSEM under the binomial model.")
  }

  if (!is.null(ct)) {
    if (!is.data.frame(ct) && !is.matrix(ct)) {
      stop("`ct` must be a data frame or matrix when provided.")
    }
    if (ncol(ct) < 2L) {
      stop("`ct` must have at least two columns: raw scores and scale scores.")
    }
    if (nrow(ct) != ni + 1L) {
      stop("`ct` must have `ni + 1` rows corresponding to raw scores 0:ni.")
    }
    # (optional but nice) check raw-score column matches 0:ni
    raw_scores <- ct[, 1]
    if (any(raw_scores != 0:ni)) {
      warning("The first column of `ct` is expected to be raw scores 0:ni.")
    }
  }

  # raw scores 0, 1, ..., ni ---------------------------------------------------
  r <- 0:ni
  m <- ni + 1L

  # pre-allocate vectors
  csemx <- numeric(m)
  csems <- if (!is.null(ct)) numeric(m) else NULL

  # loop over possible true-score values (k/ni) --------------------------------
  for (k in 0:ni) {
    idx <- k + 1L

    # treat raw score k as true score => pi_k
    pi_k <- k / ni

    # probability of each raw score under Bin(ni, pi_k)
    p <- stats::dbinom(r, size = ni, prob = pi_k)

    # SEM formula (13) in Lee's notes CH5 (raw-score metric)
    er   <- sum(r    * p)
    er2  <- sum(r^2  * p)
    csemx[idx] <- sqrt(ni / (ni - 1)) * sqrt(er2 - er^2)

    # SEM formula (14) in Lee's notes CH5 (scale metric), if ct provided
    if (!is.null(ct)) {
      s_vals <- ct[, 2]
      es    <- sum(s_vals    * p)
      es2   <- sum(s_vals^2  * p)
      csems[idx] <- sqrt(ni / (ni - 1)) * sqrt(es2 - es^2)
    }
  }

  # return ---------------------------------------------------------------------
  if (!is.null(ct)) {
    return(list(x = r, csem = csemx, cssem = csems))
  }

  return(list(x = r, csem = csemx))
}
