% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/energyGOF.R
\name{energyGOF.test}
\alias{energyGOF.test}
\alias{egof.test}
\title{Goodness-of-fit tests for univariate data via energy}
\usage{
energyGOF.test(x, y, nsim, ...)

egof.test(x, y, nsim, ...)
}
\arguments{
\item{x}{A numeric vector.}

\item{y}{A string, distribution function, or numeric vector. The
distribution to test \code{x} against.}

\item{nsim}{A non-negative integer. The number of parametric bootstrap
replicates taken to calculate the \emph{p}-value. If 0, no simulation.}

\item{...}{If \code{y} is a string or distribution function, parameters of the
distribution \code{y}. Required for a simple test. For distributions in the
\link{stats} library, parameter argument names are identical. If \code{y} is a
string, to test the \emph{composite} goodness-of-fit hypothesis that \code{x} is
distributed according to the \emph{family of distributions} \code{y}, don't pass
parameters in \code{...}. For \emph{generalized} energy tests, you can also
optionally pass the generalized energy exponent \code{pow} here. Composite
testing is not supported if \code{y} is a function. (As you can see, there is a
lot going on in \code{...} and if you don't like that, you may want to check
out \code{\link[=energyGOFdist]{energyGOFdist()}} for a structured interface.)}
}
\value{
If \code{y} is a string or function, return an object of class `htest'
representing the result of the energy goodness-of-fit hypothesis test. The
htest object has the elements:
\itemize{
\item \code{method}: Simple or Composite
\item \code{data.name}
\item \code{distribution}: The distribution object created to test
\item \code{parameter}: List of parameters if the test is simple
\item \code{nsim}: Number of bootstrap replicates
\item \code{composite_p}: TRUE/FALSE composite predicate
\item \code{statistic}: The value of the energy statistic (\eqn{Q=nE^*})
\item \code{p.value}
\item \code{sim_reps}: bootstrap replicates of energy statistic
\item \code{estimate}: Any parameter estimates, if the test is composite
}

If \code{y} is numeric, return the same htest object as \code{\link[energy:eqdist.etest]{energy::eqdist.etest()}}.
}
\description{
Perform a goodness-of-fit test of univariate data \code{x} against a
target \code{y}. \code{y} may be one of the following:
\itemize{
\item A string naming a distribution. For example, "normal". Both simple
(known parameter) and composite (unknown parameter) tests are supported,
but not all distributions allow for a composite test. See
\link{energyGOF-package} for the table of supported distributions.
\itemize{
\item Result: A parametric goodness-of-fit test is performed.
\item Allowable values: uniform, exponential, bernoulli, binomial, geometric, normal, gaussian, beta, poisson, lognormal, lnorm, laplace, doubleexponential, asymmetriclaplace, alaplace, inversegaussian, invgaussian, halfnormal, chisq, chisquared, f, gamma, weibull, cauchy, pareto.
}
\item A numeric vector of data.
\itemize{
\item Result: A two-sample, non-parametric goodness-of-fit test is performed
to test if x and y are equal in distribution.
}
\item A continuous cumulative distribution function. For example, \code{pt}.
Only simple tests are supported.
\itemize{
\item Result: \eqn{y(x)} is tested for uniformity.
}
}

\emph{P}-values are determined via parametric bootstrap. For distributions
where \eqn{E|Y|} is not finite (Cauchy, Pareto), a \emph{generalized} energy
goodness-of-fit test is performed, and an additional tuning parameter
\code{pow} is required.
}
\examples{
x <- rnorm(10)
y <- rt(10, 4)

## Composite energy goodness-of-fit test (test for Normality with unknown
## parameters)

energyGOF.test(x, "normal", nsim = 10)

## Simple energy goodness-of-fit test (test for Normality with known
## parameters). egof.test is an alias for energyGOF.test.

egof.test(x, "normal", nsim = 10, mean = 0, sd = 1)

## Alternatively, use the energyGOFdist generic directly so that you do not need
## to pass parameter names into `...`

energyGOFdist(x, normal_dist(0, 1), nsim = 10)

## Conduct a two-sample test

egof.test(x, y, 0)

## Conduct a test against any continuous distribution function

egof.test(x, pcauchy, 0)

## Simple energy goodness-of-fit test for Weibull distribution

y <- rweibull(10, 1, 1)
energyGOF.test(y, "weibull", shape = 1, scale = 3, nsim = 10)

## Alternatively, use the energyGOFdist generic directly, which is slightly less
## verbose. egofd is an alias for energyGOFdist.

egofd(y, weibull_dist(1, 3), nsim = 10)

## Conduct a generalized GOF test. `pow` is the exponent *s* in the generalized
## energy statistic. Pow is only necessary when testing Cauchy, and
## Pareto distributions. If you don't set a pow, there is a default for each
## of the distributions, but the default isn't necessarily better than any
## other number.

egofd(rcauchy(100),
   cauchy_dist(location = 0, scale = 1, pow = 0.5),
   nsim = 10)

## energyGOF does not support tests with a mix of known and unknown
## parameters, so this will result in an error.

\donttest{
  energyGOF.test(x, "normal", mean = 0, nsim = 10) # sd is missing
}


}
\seealso{
\itemize{
\item \link{energyGOF-package} for specifics on the distributions available to test.
\item \code{\link[=energyGOFdist]{energyGOFdist()}} for the alternate S3 interface for parametric testing.
\item \link[stats]{Distributions} for a list of distributions available
in most R installations.
\item \code{\link[energy:eqdist.etest]{energy::eqdist.etest()}} for information on the two-sample test.
\item \code{\link[energy:normalGOF]{energy::normal.test()}} for the energy goodness-of-fit test with unknown
parameters. The tests for (multivariate) Normal in the energy package are
implemented with compiled code, and are faster than the one available in the
energyGOF package.
\item \code{\link[energy:poisson]{energy::poisson.mtest()}} for a different Poisson goodness-of-fit test
based on mean distances.
}
}
\author{
John T. Haman
}
