% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/arfima.R
\name{ARFIMA}
\alias{ARFIMA}
\alias{report.fbl_ARFIMA}
\title{Estimate an ARFIMA model}
\usage{
ARFIMA(
  formula,
  ic = c("aicc", "aic", "bic"),
  selection_metric = function(x) x[[ic]],
  stepwise = TRUE,
  greedy = TRUE,
  order_constraint = p + q <= 6,
  trace = FALSE,
  ...
)
}
\arguments{
\item{formula}{Model specification (see "Specials" section).}

\item{ic}{The information criterion used in selecting the model.}

\item{selection_metric}{A function used to compute a metric from the fitted
object which is minimised to select the best model.}

\item{stepwise, greedy, order_constraint, trace}{Arguments kept for API
compatibility with \code{ARIMA()}. Currently not fully implemented for ARFIMA.}

\item{...}{Further arguments passed to \code{\link[fracdiff:fracdiff]{fracdiff::fracdiff()}}.}
}
\value{
A model specification.
}
\description{
Searches through the model space specified in the specials to identify a
suitable ARFIMA model. ARFIMA (AutoRegressive Fractionally Integrated Moving
Average) models extend ARIMA models by allowing fractional differencing,
which is useful for modeling long memory processes. The model is implemented
using \code{\link[fracdiff:fracdiff]{fracdiff::fracdiff()}} and allows ARFIMA models to be used in the
fable framework.
}
\section{Parameterisation}{


An ARFIMA(p,d,q) model is defined as:

\deqn{(1-\phi_1B - \cdots - \phi_p B^p)(1-B)^d (y_t - \mu) = (1 + \theta_1 B + \cdots + \theta_q B^q)\varepsilon_t}

where \eqn{\mu} is the mean of the series, and \eqn{d} can take fractional
values (typically between -0.5 and 0.5), allowing the model to capture long
memory behavior. When \eqn{d} is an integer, the model reduces to a standard
ARIMA model.

\strong{Note:} This uses a mean form parameterisation where the data is de-meaned
before fitting. This differs from \code{\link[=ARIMA]{ARIMA()}} which uses a constant form
parameterisation.

The fractional differencing operator \eqn{(1-B)^d} is computed using the
fast algorithm of Jensen and Nielsen (2014), which is implemented in the
fracdiff package.
}

\section{Specials}{


The \emph{specials} define the space over which \code{ARFIMA} will search for the
model that best fits the data. If the RHS of \code{formula} is left blank, the
default search space is given by \code{pdq()}: a model with candidate
non-seasonal terms and fractional differencing, but no exogenous regressors.

Note that ARFIMA does not support seasonal differencing (PDQ terms). For
seasonal data, consider using \code{\link[=ARIMA]{ARIMA()}} instead, or pre-process your data
to remove seasonality.

\subsection{pdq}{
The \code{pdq} special is used to specify the components of the ARFIMA model.
\preformatted{
pdq(p = 0:5, d = NULL, q = 0:5, 
    d_range = c(0, 0.5),
    p_init = 2, q_init = 2, fixed = list())
}

\tabular{ll}{
\code{p}       \tab The order of the auto-regressive (AR) terms. If multiple values are provided, the one which minimises \code{ic} will be chosen. \cr
\code{d}       \tab The fractional differencing parameter. If \code{NULL} (default), it will be estimated. If a single numeric value is provided, it will be held fixed at that value. Unlike ARIMA, only a single value or \code{NULL} is allowed. \cr
\code{q}       \tab The order of the moving average (MA) terms. If multiple values are provided, the one which minimises \code{ic} will be chosen. \cr
\code{d_range} \tab A numeric vector of length 2 specifying the range for estimating \code{d}. Only used when \code{d = NULL}. Typical values are between -0.5 and 0.5. \cr
\code{p_init}  \tab If \code{stepwise = TRUE}, \code{p_init} provides the initial value for \code{p} for the stepwise search procedure. \cr
\code{q_init}  \tab If \code{stepwise = TRUE}, \code{q_init} provides the initial value for \code{q} for the stepwise search procedure. \cr
\code{fixed}   \tab A named list of fixed parameters for coefficients. The names identify the coefficient, beginning with either \code{ar} or \code{ma}, followed by the lag order. For example, \code{fixed = list(ar1 = 0.3, ma2 = 0)}.
}
}

\subsection{xreg}{
Exogenous regressors can be included in an ARFIMA model without explicitly
using the \code{xreg()} special. Common exogenous regressor specials as specified
in \code{\link{common_xregs}} can also be used. These regressors are handled using
\code{\link[stats:model.frame]{stats::model.frame()}}, and so interactions and other functionality behaves
similarly to \code{\link[stats:lm]{stats::lm()}}.

The inclusion of a constant in the model follows similar rules to
\code{\link[stats:lm]{stats::lm()}}, where including \code{1} will add a constant and \code{0} or \code{-1}
will remove the constant. If left out, the inclusion of a constant will be
determined by minimising \code{ic}.

\preformatted{
xreg(..., fixed = list())
}

\tabular{ll}{
\code{...}   \tab Bare expressions for the exogenous regressors (such as \code{log(x)}) \cr
\code{fixed} \tab A named list of fixed parameters for coefficients. The names identify the coefficient, and should match the name of the regressor. For example, \code{fixed = list(constant = 20)}.
}
}
}

\examples{
\dontshow{if (requireNamespace("fracdiff", quietly = TRUE)) withAutoprint(\{ # examplesIf}
library(tsibble)
library(dplyr)

# Automatic ARFIMA specification
as_tsibble(sunspot.year) \%>\%
 model(arfima = ARFIMA(value)) \%>\%
 report()
 
# Manual ARFIMA specification with fixed d
as_tsibble(sunspot.year) \%>\%
  model(arfima = ARFIMA(value ~ pdq(p = 1, d = 0.3, q = 1))) \%>\%
  report()
\dontshow{\}) # examplesIf}
}
\references{
Jensen, A. N. and Nielsen, M. Ø. (2014) A Fast Fractional Difference
Algorithm. Journal of Time Series Analysis 35(5), 428–436.
\doi{10.1111/jtsa.12074}
}
\seealso{
\code{\link[=ARIMA]{ARIMA()}} for standard ARIMA models with integer differencing.

\href{https://otexts.com/fpp3/arima.html}{Forecasting: Principles and Practices, ARIMA models (chapter 9)}

\code{\link[fracdiff:fracdiff]{fracdiff::fracdiff()}} for the underlying fitting function.
}
