% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gpb.train.R
\name{gpb.train}
\alias{gpb.train}
\title{Main training logic for GBPoost}
\usage{
gpb.train(params = list(), data, nrounds = 100L, gp_model = NULL,
  use_gp_model_for_validation = TRUE, train_gp_model_cov_pars = TRUE,
  valids = list(), obj = NULL, eval = NULL, verbose = 1L,
  record = TRUE, eval_freq = 1L, init_model = NULL, colnames = NULL,
  categorical_feature = NULL, early_stopping_rounds = NULL,
  callbacks = list(), reset_data = FALSE, ...)
}
\arguments{
\item{params}{list of "tuning" parameters. 
See \href{https://github.com/fabsig/GPBoost/blob/master/docs/Parameters.rst}{the parameter documentation} for more information. 
A few key parameters:
           \itemize{
               \item{\code{learning_rate}: The learning rate, also called shrinkage or damping parameter 
               (default = 0.1). An important tuning parameter for boosting. Lower values usually 
               lead to higher predictive accuracy but more boosting iterations are needed }
               \item{\code{num_leaves}: Number of leaves in a tree. Tuning parameter for 
               tree-boosting (default = 31)}
               \item{\code{max_depth}: Maximal depth of a tree. Tuning parameter for tree-boosting (default = no limit)}
               \item{\code{min_data_in_leaf}: Minimal number of samples per leaf. Tuning parameter for 
               tree-boosting (default = 20)}
               \item{\code{lambda_l2}: L2 regularization (default = 0)}
               \item{\code{lambda_l1}: L1 regularization (default = 0)}
               \item{\code{max_bin}: Maximal number of bins that feature values will be bucketed in (default = 255)}
               \item{\code{line_search_step_length} (default = FALSE): If TRUE, a line search is done to find the optimal 
               step length for every boosting update (see, e.g., Friedman 2001). This is then multiplied by the learning rate }
               \item{\code{train_gp_model_cov_pars} (default = TRUE): If TRUE, the covariance parameters of the Gaussian process 
               are estimated in every boosting iterations,  otherwise the gp_model parameters are not estimated. 
               In the latter case, you need to either estimate them beforehand or provide values via 
               the 'init_cov_pars' parameter when creating the gp_model }
               \item{\code{use_gp_model_for_validation} (default = TRUE): If TRUE, the Gaussian process is also used 
               (in addition to the tree model) for calculating predictions on the validation data }
               \item{\code{leaves_newton_update} (default = FALSE): Set this to TRUE to do a Newton update step for the tree leaves 
               after the gradient step. Applies only to Gaussian process boosting (GPBoost algorithm) }
               \item{num_threads: Number of threads. For the best speed, set this to
                            the number of real CPU cores(\code{parallel::detectCores(logical = FALSE)}),
                            not the number of threads (most CPU using hyper-threading to generate 2 threads
                            per CPU core).}
           }}

\item{data}{a \code{gpb.Dataset} object, used for training. Some functions, such as \code{\link{gpb.cv}},
may allow you to pass other types of data like \code{matrix} and then separately supply
\code{label} as a keyword argument.}

\item{nrounds}{number of boosting iterations (= number of trees). This is the most important tuning parameter for boosting}

\item{gp_model}{A \code{GPModel} object that contains the random effects (Gaussian process and / or grouped random effects) model}

\item{use_gp_model_for_validation}{Boolean. If TRUE, the \code{gp_model} 
(Gaussian process and/or random effects) is also used (in addition to the tree model) for calculating 
predictions on the validation data. If FALSE, the \code{gp_model} (random effects part) is ignored 
for making predictions and only the tree ensemble is used for making predictions for calculating the validation / test error.}

\item{train_gp_model_cov_pars}{Boolean. If TRUE, the covariance parameters 
of the \code{gp_model} (Gaussian process and/or random effects) are estimated in every 
boosting iterations, otherwise the \code{gp_model} parameters are not estimated. 
In the latter case, you need to either estimate them beforehand or provide the values via 
the \code{init_cov_pars} parameter when creating the \code{gp_model}}

\item{valids}{a list of \code{gpb.Dataset} objects, used for validation}

\item{obj}{(character) The distribution of the response variable (=label) conditional on fixed and random effects.
This only needs to be set when doing independent boosting without random effects / Gaussian processes.}

\item{eval}{Evaluation metric to be monitored when doing CV and parameter tuning. 
This can be a string, function, or list with a mixture of strings and functions.

            \itemize{
                \item{\bold{a. character vector}:
                    Non-exhaustive list of supported metrics: "test_neg_log_likelihood", "mse", "rmse", "mae", 
                    "auc", "average_precision", "binary_logloss", "binary_error"
                    See \href{https://gpboost.readthedocs.io/en/latest/Parameters.html#metric-parameters}{
                    the "metric" section of the parameter documentation}
                    for a complete list of valid metrics.
                }
                \item{\bold{b. function}:
                     You can provide a custom evaluation function. This
                     should accept the keyword arguments \code{preds} and \code{dtrain} and should return a named
                     list with three elements:
                     \itemize{
                         \item{\code{name}: A string with the name of the metric, used for printing
                             and storing results.
                         }
                         \item{\code{value}: A single number indicating the value of the metric for the
                             given predictions and true values
                         }
                         \item{
                             \code{higher_better}: A boolean indicating whether higher values indicate a better fit.
                             For example, this would be \code{FALSE} for metrics like MAE or RMSE.
                         }
                     }
                }
                \item{\bold{c. list}:
                    If a list is given, it should only contain character vectors and functions.
                    These should follow the requirements from the descriptions above.
                }
            }}

\item{verbose}{verbosity for output, if <= 0, also will disable the print of evaluation during training}

\item{record}{Boolean, TRUE will record iteration message to \code{booster$record_evals}}

\item{eval_freq}{evaluation output frequency, only effect when verbose > 0}

\item{init_model}{path of model file of \code{gpb.Booster} object, will continue training from this model}

\item{colnames}{feature names, if not null, will use this to overwrite the names in dataset}

\item{categorical_feature}{categorical features. This can either be a character vector of feature
names or an integer vector with the indices of the features (e.g.
\code{c(1L, 10L)} to say "the first and tenth columns").}

\item{early_stopping_rounds}{int. Activates early stopping. Requires at least one validation data
and one metric. When this parameter is non-null,
training will stop if the evaluation of any metric on any validation set
fails to improve for \code{early_stopping_rounds} consecutive boosting rounds.
If training stops early, the returned model will have attribute \code{best_iter}
set to the iteration number of the best iteration.}

\item{callbacks}{List of callback functions that are applied at each iteration.}

\item{reset_data}{Boolean, setting it to TRUE (not the default value) will transform the
booster model into a predictor model which frees up memory and the
original datasets}

\item{...}{other parameters, see \href{https://github.com/fabsig/GPBoost/blob/master/docs/Parameters.rst}{the parameter documentation} for more information.}
}
\value{
a trained booster model \code{gpb.Booster}.
}
\description{
Logic to train with GBPoost
}
\section{Early Stopping}{


         "early stopping" refers to stopping the training process if the model's performance on a given
         validation set does not improve for several consecutive iterations.

         If multiple arguments are given to \code{eval}, their order will be preserved. If you enable
         early stopping by setting \code{early_stopping_rounds} in \code{params}, by default all
         metrics will be considered for early stopping.

         If you want to only consider the first metric for early stopping, pass
         \code{first_metric_only = TRUE} in \code{params}. Note that if you also specify \code{metric}
         in \code{params}, that metric will be considered the "first" one. If you omit \code{metric},
         a default metric will be used based on your choice for the parameter \code{obj} (keyword argument)
         or \code{objective} (passed into \code{params}).
}

\examples{
# See https://github.com/fabsig/GPBoost/tree/master/R-package for more examples

\donttest{
library(gpboost)
data(GPBoost_data, package = "gpboost")

#--------------------Combine tree-boosting and grouped random effects model----------------
# Create random effects model
gp_model <- GPModel(group_data = group_data[,1], likelihood = "gaussian")
# The default optimizer for covariance parameters (hyperparameters) is 
# Nesterov-accelerated gradient descent.
# This can be changed to, e.g., Nelder-Mead as follows:
# re_params <- list(optimizer_cov = "nelder_mead")
# gp_model$set_optim_params(params=re_params)
# Use trace = TRUE to monitor convergence:
# re_params <- list(trace = TRUE)
# gp_model$set_optim_params(params=re_params)
dtrain <- gpb.Dataset(data = X, label = y)
# Train model
bst <- gpb.train(data = dtrain, gp_model = gp_model, nrounds = 16,
                 learning_rate = 0.05, max_depth = 6, min_data_in_leaf = 5,
                 verbose = 0)
# Estimated random effects model
summary(gp_model)
# Make predictions
pred <- predict(bst, data = X_test, group_data_pred = group_data_test[,1],
                predict_var= TRUE)
pred$random_effect_mean # Predicted mean
pred$random_effect_cov # Predicted variances
pred$fixed_effect # Predicted fixed effect from tree ensemble
# Sum them up to otbain a single prediction
pred$random_effect_mean + pred$fixed_effect

#--------------------Combine tree-boosting and Gaussian process model----------------
# Create Gaussian process model
gp_model <- GPModel(gp_coords = coords, cov_function = "exponential",
                    likelihood = "gaussian")
# Train model
dtrain <- gpb.Dataset(data = X, label = y)
bst <- gpb.train(data = dtrain, gp_model = gp_model, nrounds = 16,
                 learning_rate = 0.05, max_depth = 6, min_data_in_leaf = 5,
                 verbose = 0)
# Estimated random effects model
summary(gp_model)
# Make predictions
pred <- predict(bst, data = X_test, gp_coords_pred = coords_test,
                predict_cov_mat =TRUE)
pred$random_effect_mean # Predicted (posterior) mean of GP
pred$random_effect_cov # Predicted (posterior) covariance matrix of GP
pred$fixed_effect # Predicted fixed effect from tree ensemble
# Sum them up to otbain a single prediction
pred$random_effect_mean + pred$fixed_effect


#--------------------Using validation data-------------------------
set.seed(1)
train_ind <- sample.int(length(y),size=250)
dtrain <- gpb.Dataset(data = X[train_ind,], label = y[train_ind])
dtest <- gpb.Dataset.create.valid(dtrain, data = X[-train_ind,], label = y[-train_ind])
valids <- list(test = dtest)
gp_model <- GPModel(group_data = group_data[train_ind,1], likelihood="gaussian")
# Need to set prediction data for gp_model
gp_model$set_prediction_data(group_data_pred = group_data[-train_ind,1])
# Training with validation data and use_gp_model_for_validation = TRUE
bst <- gpb.train(data = dtrain, gp_model = gp_model, nrounds = 100,
                 learning_rate = 0.05, max_depth = 6, min_data_in_leaf = 5,
                 verbose = 1, valids = valids,
                 early_stopping_rounds = 10, use_gp_model_for_validation = TRUE)
print(paste0("Optimal number of iterations: ", bst$best_iter,
             ", best test error: ", bst$best_score))
# Plot validation error
val_error <- unlist(bst$record_evals$test$l2$eval)
plot(1:length(val_error), val_error, type="l", lwd=2, col="blue",
     xlab="iteration", ylab="Validation error", main="Validation error vs. boosting iteration")


#--------------------Do Newton updates for tree leaves---------------
# Note: run the above examples first
bst <- gpb.train(data = dtrain, gp_model = gp_model, nrounds = 100,
                 learning_rate = 0.05, max_depth = 6, min_data_in_leaf = 5,
                 verbose = 1, valids = valids,
                 early_stopping_rounds = 5, use_gp_model_for_validation = FALSE,
                 leaves_newton_update = TRUE)
print(paste0("Optimal number of iterations: ", bst$best_iter,
             ", best test error: ", bst$best_score))
# Plot validation error
val_error <- unlist(bst$record_evals$test$l2$eval)
plot(1:length(val_error), val_error, type="l", lwd=2, col="blue",
     xlab="iteration", ylab="Validation error", main="Validation error vs. boosting iteration")


#--------------------GPBoostOOS algorithm: GP parameters estimated out-of-sample----------------
# Create random effects model and dataset
gp_model <- GPModel(group_data = group_data[,1], likelihood="gaussian")
dtrain <- gpb.Dataset(X, label = y)
params <- list(learning_rate = 0.05,
               max_depth = 6,
               min_data_in_leaf = 5)
# Stage 1: run cross-validation to (i) determine to optimal number of iterations
#           and (ii) to estimate the GPModel on the out-of-sample data
cvbst <- gpb.cv(params = params,
                data = dtrain,
                gp_model = gp_model,
                nrounds = 100,
                nfold = 4,
                eval = "l2",
                early_stopping_rounds = 5,
                use_gp_model_for_validation = TRUE,
                fit_GP_cov_pars_OOS = TRUE)
print(paste0("Optimal number of iterations: ", cvbst$best_iter))
# Estimated random effects model
# Note: ideally, one would have to find the optimal combination of
#               other tuning parameters such as the learning rate, tree depth, etc.)
summary(gp_model)
# Stage 2: Train tree-boosting model while holding the GPModel fix
bst <- gpb.train(data = dtrain,
                 gp_model = gp_model,
                 nrounds = cvbst$best_iter,
                 learning_rate = 0.05,
                 max_depth = 6,
                 min_data_in_leaf = 5,
                 verbose = 0,
                 train_gp_model_cov_pars = FALSE)
# The GPModel has not changed:
summary(gp_model)
}
}
\author{
Fabio Sigrist, authors of the LightGBM R package
}
