#' Find parent sets from the candidate cultivars
#' @param gene_df1 Data frame of one set of haplotype. Values take 1 (target allele) or 0 (non-target).
#' @param gene_df2 Data frame of the other set of haplotype. Values take 1 or 0.
#' @param line_id character vector of cultivar names
#'
#' @importFrom utils combn
#'
#' @returns
#' `findPset()` returns a list of parental cultivar sets.
#' Each element of list contains the parent names.
#'
#' @examples
#' gene_df1 <-
#'  data.frame(x1 = c(1, 1, 1, 1, 1, 0, 0),
#'             x2 = c(1, 1, 1, 0, 1, 1, 0),
#'             x3 = c(1, 1, 1, 0, 1, 1, 0),
#'             x4 = c(1, 1, 0, 0, 0, 0, 0),
#'             x5 = c(0, 0, 1, 0, 1, 1, 0),
#'             x6 = c(0, 0, 1, 1, 0, 0, 0),
#'             x7 = c(0, 1, 1, 0, 0, 0, 1))

#' gene_df2 <-
#'  data.frame(x1 = c(0, 0, 0, 0, 0, 0, 0),
#'             x2 = c(0, 0, 0, 0, 0, 1, 0),
#'             x3 = c(1, 0, 0, 0, 0, 0, 0),
#'             x4 = c(1, 1, 0, 0, 0, 0, 0),
#'             x5 = c(0, 0, 1, 0, 1, 1, 0),
#'             x6 = c(0, 0, 1, 1, 0, 0, 0),
#'             x7 = c(0, 1, 1, 0, 0, 0, 1))
#'
#' line_id <- c("x1", "x2", "x3", "x4", "x5", "x6", "x7")
#'
#' findPset(gene_df1, gene_df2, line_id)
#'
#' @export

findPset <- function(gene_df1, gene_df2, line_id){
  n_line <- length(line_id)
  line_comb_lis <- NULL

  for(i in 1:n_line){
    comb_all <- combn(line_id, i)
    n_comb <- ncol(comb_all)
    for(j in 1:n_comb){
      line_comb_id <- comb_all[, j]
      gene_df12 <- cbind(gene_df1[, line_comb_id], gene_df2[, line_comb_id])
      gene_id <- apply(gene_df12, 1, FUN = function(x){sum(x == 1) > 0})

      if(!(F %in% gene_id)){
        line_comb_lis <- addLineComb(line_comb_lis, line_comb_id)
      }
    }
  }
  return(line_comb_lis)
}

addLineComb <- function(line_comb_lis, line_comb_id){
  check_set <- lapply(line_comb_lis, FUN = setdiff, line_comb_id)
  check_TF <- unlist(lapply(check_set, length))
  if(!(0 %in% check_TF)){
    line_comb_lis <- c(line_comb_lis, list(line_comb_id))
  }
  return(line_comb_lis)
}
