#' Function to get one crossing scheme from all the crossing schemes.
#' @param gpyramid_all_obj gpyramid_all object.
#' @param cross_id cross_id in cost_all data frame.
#'
#' @returns getFromAll function returns a `getFromAll` object.
#' This is a named list with the following components:
#' * `topolo` (`phylo`): crossing scheme described as the phylo object, which is defined by ape package.
#' * `last_cross` (`boolean`): Whether or not to conduct the last cross to a cultivar without target alleles.
#' * `last_selfint` (`boolean`): Whether or not to conduct the last selfing.
#' * `cost_onecrossing` (`list`): List of output information about one gene pyramiding scheme.
#' The object has class `getFromAll` and can be summarized using `summary()`.
#' @export

getFromAll <-
  function(gpyramid_all_obj, cross_id){

    cost_all <- gpyramid_all_obj$cost_all
    line_comb_lis <- gpyramid_all_obj$line_comb_lis
    gene_df1 <- gpyramid_all_obj$gene_df1
    gene_df2 <- gpyramid_all_obj$gene_df2
    recom_mat <- gpyramid_all_obj$recom_mat
    prob_total <- gpyramid_all_obj$prob_total
    last_cross <- gpyramid_all_obj$last_cross
    last_selfing <- gpyramid_all_obj$last_selfing

    line_comb_id <- cost_all$line_comb_id[cost_all$cross_id == cross_id]
    TR_id <- cost_all$TR_id[cost_all$cross_id == cross_id]

    line_comb <- line_comb_lis[[line_comb_id]]
    n_line <- length(line_comb)
    TR <- allCrosses(n_line, line_comb)

    # prepare genotypes of parents
    gene_df1_sel <- gene_df1[, line_comb]
    gene_df2_sel <- gene_df2[, line_comb]

    # Calculate cost
    topolo <- TR[[TR_id]]
    cost_onecrossing <- calcCost(topolo, gene_df1_sel, gene_df2_sel, recom_mat, prob_total, last_cross, last_selfing)

    output <- list(topolo = topolo,
                   last_cross = last_cross,
                   last_selfing = last_selfing,
                   cost_onecrossing = cost_onecrossing)

    class(output) <- "getFromAll"

    return(output)
}

#' @method summary getFromAll
#' @export

summary.getFromAll <- function(object, ...){
  n_plant_df <- object$cost_onecrossing$n_plant_df
  n_plant_df$n_plant <- ceiling(n_plant_df$n_plant)
  n_plant_df$nodeid <- as.character(n_plant_df$nodeid)

  n_int_node <- nrow(n_plant_df) - object$last_cross - object$last_selfing
  int_node_id <- 1:n_int_node
  n_plant_df$nodeid[int_node_id] <- paste("node_", n_plant_df$nodeid[int_node_id], sep = "")

  if(object$last_cross & object$last_selfing){
    n_plant_df$nodeid[n_int_node + 1] <- "last cross"
    n_plant_df$nodeid[n_int_node + 2] <- "last selfing"
  }else if(object$last_cross){
    n_plant_df$nodeid[n_int_node + 1] <- "last cross"
  }else if(object$last_selfing){
    n_plant_df$nodeid[n_int_node + 1] <- "last selfing"
  }

  n_tip <- min(object$cost_onecrossing$n_plant_df$nodeid) - 1
  gene_df1_sel <- object$cost_onecrossing$gene_df1_sel
  gene_df2_sel <- object$cost_onecrossing$gene_df2_sel

  cat("Number of necessary plants for each crossing\n\n")
  print(n_plant_df[, c("nodeid", "n_plant")])

  cat("\nTarget haplotype of crossing\n")

  for(i in (n_tip + 1):(n_tip + n_int_node)){
    nodeid_one <- colnames(gene_df1_sel)[i]
    cat(paste("\nnodeid=", nodeid_one, "\n", sep = ""))
    target_haplo <- data.frame(hap1 = gene_df1_sel[,i],
                               hap2 = gene_df2_sel[,i])
    print(target_haplo)
  }
}


