/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5EAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5EApkg.h"     
#include "H5FLprivate.h" 

H5FL_DEFINE_STATIC(H5EA_dblk_page_t);

H5EA_dblk_page_t *
H5EA__dblk_page_alloc(H5EA_hdr_t *hdr, H5EA_sblock_t *parent)
{
    H5EA_dblk_page_t *dblk_page = NULL; 
    H5EA_dblk_page_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (NULL == (dblk_page = H5FL_CALLOC(H5EA_dblk_page_t)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array data block page");

    
    if (H5EA__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINC, NULL, "can't increment reference count on shared array header");
    dblk_page->hdr = hdr;

    
    dblk_page->parent = parent;

    
    if (NULL == (dblk_page->elmts = H5EA__hdr_alloc_elmts(hdr, hdr->dblk_page_nelmts)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for data block page element buffer");

    
    ret_value = dblk_page;

done:
    if (!ret_value)
        if (dblk_page && H5EA__dblk_page_dest(dblk_page) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array data block page");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__dblk_page_create(H5EA_hdr_t *hdr, H5EA_sblock_t *parent, haddr_t addr)
{
    H5EA_dblk_page_t *dblk_page = NULL;  
    bool              inserted  = false; 
    herr_t            ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (NULL == (dblk_page = H5EA__dblk_page_alloc(hdr, parent)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, FAIL,
                    "memory allocation failed for extensible array data block page");

    
    dblk_page->addr = addr;
    dblk_page->size = H5EA_DBLK_PAGE_SIZE(hdr);

    
    if ((hdr->cparam.cls->fill)(dblk_page->elmts, (size_t)hdr->dblk_page_nelmts) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, FAIL,
                    "can't set extensible array data block page elements to class's fill value");

    
    if (H5AC_insert_entry(hdr->f, H5AC_EARRAY_DBLK_PAGE, dblk_page->addr, dblk_page, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINSERT, FAIL, "can't add extensible array data block page to cache");
    inserted = true;

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, dblk_page) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, FAIL,
                        "unable to add extensible array entry as child of array proxy");
        dblk_page->top_proxy = hdr->top_proxy;
    } 

done:
    if (ret_value < 0)
        if (dblk_page) {
            
            if (inserted)
                if (H5AC_remove_entry(dblk_page) < 0)
                    HDONE_ERROR(H5E_EARRAY, H5E_CANTREMOVE, FAIL,
                                "unable to remove extensible array data block page from cache");

            
            if (H5EA__dblk_page_dest(dblk_page) < 0)
                HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, FAIL,
                            "unable to destroy extensible array data block page");
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5EA_dblk_page_t *
H5EA__dblk_page_protect(H5EA_hdr_t *hdr, H5EA_sblock_t *parent, haddr_t dblk_page_addr, unsigned flags)
{
    H5EA_dblk_page_t         *dblk_page = NULL; 
    H5EA_dblk_page_cache_ud_t udata;            
    H5EA_dblk_page_t         *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(dblk_page_addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.hdr            = hdr;
    udata.parent         = parent;
    udata.dblk_page_addr = dblk_page_addr;

    
    if (NULL == (dblk_page = (H5EA_dblk_page_t *)H5AC_protect(hdr->f, H5AC_EARRAY_DBLK_PAGE, dblk_page_addr,
                                                              &udata, flags)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, NULL,
                    "unable to protect extensible array data block page, address = %llu",
                    (unsigned long long)dblk_page_addr);

    
    if (hdr->top_proxy && NULL == dblk_page->top_proxy) {
        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, dblk_page) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, NULL,
                        "unable to add extensible array entry as child of array proxy");
        dblk_page->top_proxy = hdr->top_proxy;
    } 

    
    ret_value = dblk_page;

done:
    
    if (!ret_value) {
        
        if (dblk_page &&
            H5AC_unprotect(hdr->f, H5AC_EARRAY_DBLK_PAGE, dblk_page->addr, dblk_page, H5AC__NO_FLAGS_SET) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, NULL,
                        "unable to unprotect extensible array data block page, address = %llu",
                        (unsigned long long)dblk_page->addr);
    } 
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__dblk_page_unprotect(H5EA_dblk_page_t *dblk_page, unsigned cache_flags)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblk_page);

    
    if (H5AC_unprotect(dblk_page->hdr->f, H5AC_EARRAY_DBLK_PAGE, dblk_page->addr, dblk_page, cache_flags) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL,
                    "unable to unprotect extensible array data block page, address = %llu",
                    (unsigned long long)dblk_page->addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__dblk_page_dest(H5EA_dblk_page_t *dblk_page)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(dblk_page);
    assert(!dblk_page->has_hdr_depend);

    
    if (dblk_page->hdr) {
        
        if (dblk_page->elmts) {
            
            if (H5EA__hdr_free_elmts(dblk_page->hdr, dblk_page->hdr->dblk_page_nelmts, dblk_page->elmts) < 0)
                HGOTO_ERROR(H5E_EARRAY, H5E_CANTFREE, FAIL,
                            "unable to free extensible array data block element buffer");
            dblk_page->elmts = NULL;
        } 

        
        if (H5EA__hdr_decr(dblk_page->hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared array header");
        dblk_page->hdr = NULL;
    } 

    
    assert(NULL == dblk_page->top_proxy);

    
    dblk_page = H5FL_FREE(H5EA_dblk_page_t, dblk_page);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
