/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5E_FRIEND      
#include "H5TSmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Epkg.h"      
#include "H5TSpkg.h"     

#ifdef H5_HAVE_THREADSAFE_API

#ifdef H5_HAVE_C11_THREADS
#define H5TS_ONCE_INIT_FUNC H5TS__c11_first_thread_init
#else
#ifdef H5_HAVE_WIN_THREADS
#define H5TS_ONCE_INIT_FUNC H5TS__win32_process_enter
#else
#define H5TS_ONCE_INIT_FUNC H5TS__pthread_first_thread_init
#endif 
#endif

typedef struct H5TS_thread_info_t {
    uint64_t            id;               
    struct H5CX_node_t *api_ctx_node_ptr; 
    H5E_stack_t         err_stack;        
#ifdef H5_HAVE_CONCURRENCY
    unsigned dlftt; 
#endif              
} H5TS_thread_info_t;

typedef struct H5TS_tinfo_node_t {
    struct H5TS_tinfo_node_t *next;
    H5TS_thread_info_t        info;
} H5TS_tinfo_node_t;

static H5TS_tinfo_node_t *H5TS__tinfo_create(void);
#ifdef H5_HAVE_CONCURRENCY
static herr_t H5TS__get_dlftt(unsigned *dlftt);
static herr_t H5TS__set_dlftt(unsigned dlftt);
static herr_t H5TS__inc_dlftt(void);
static herr_t H5TS__dec_dlftt(void);
#endif 

bool H5_PKG_INIT_VAR = false;

H5TS_key_t H5TS_thrd_info_key_g;

static H5TS_once_t H5TS_first_init_s = H5TS_ONCE_INITIALIZER;

static H5TS_tinfo_node_t *H5TS_tinfo_next_free_s = NULL;
static uint64_t           H5TS_next_thrd_id_s    = 0;

static H5TS_mutex_t H5TS_tinfo_mtx_s;

herr_t
H5TS__init_package(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
#ifdef H5_HAVE_THREADSAFE
    if (H5_UNLIKELY(H5TS_mutex_init(&H5TS_api_info_p.api_mutex, H5TS_MUTEX_TYPE_RECURSIVE) < 0))
        HGOTO_DONE(FAIL);
    H5TS_api_info_p.lock_count = 0;
#else 
    if (H5_UNLIKELY(H5TS_rwlock_init(&H5TS_api_info_p.api_lock) < 0))
        HGOTO_DONE(FAIL);
#endif
    H5TS_atomic_init_uint(&H5TS_api_info_p.attempt_lock_count, 0);

    
    if (H5_UNLIKELY(H5TS__tinfo_init() < 0))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5TS_term_package(void)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
#ifdef H5_HAVE_THREADSAFE
    H5TS_mutex_destroy(&H5TS_api_info_p.api_mutex);
#else 
    H5TS_rwlock_destroy(&H5TS_api_info_p.api_lock);
#endif
    H5TS_atomic_destroy_uint(&H5TS_api_info_p.attempt_lock_count);

    FUNC_LEAVE_NOAPI_VOID
} 

#ifdef H5_HAVE_CONCURRENCY

herr_t
H5TS_user_cb_prepare(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5TS__inc_dlftt() < 0)
        HGOTO_ERROR(H5E_LIB, H5E_CANTINC, FAIL, "unable to increment DLFTT value");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5TS_user_cb_restore(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5TS__dec_dlftt() < 0)
        HGOTO_ERROR(H5E_LIB, H5E_CANTDEC, FAIL, "unable to decrement DLFTT value");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

herr_t
H5TS__api_mutex_acquire(unsigned lock_count, bool *acquired)
{
#ifdef H5_HAVE_CONCURRENCY
    unsigned dlftt = 0;
#endif
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

#ifdef H5_HAVE_THREADSAFE
    
    if (H5_UNLIKELY(H5TS_mutex_trylock(&H5TS_api_info_p.api_mutex, acquired) < 0))
        HGOTO_DONE(FAIL);

    
    if (*acquired) {
        for (unsigned u = 0; u < (lock_count - 1); u++)
            if (H5_UNLIKELY(H5TS_mutex_lock(&H5TS_api_info_p.api_mutex) < 0))
                HGOTO_DONE(FAIL);
        H5TS_api_info_p.lock_count += lock_count;
    }
#else 
    
    if (H5_UNLIKELY(H5TS__get_dlftt(&dlftt) < 0))
        HGOTO_DONE(FAIL);

    
    if (0 == dlftt) {
        
        if (H5_UNLIKELY(H5TS_rwlock_trywrlock(&H5TS_api_info_p.api_lock, acquired) < 0))
            HGOTO_DONE(FAIL);
    }
    else
        *acquired = true;

    
    if (*acquired)
        
        if (H5_UNLIKELY(H5TS__set_dlftt(dlftt + lock_count) < 0))
            HGOTO_DONE(FAIL);
#endif

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

#ifdef H5_HAVE_THREADSAFE
herr_t
H5TS_api_lock(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(!H5_INIT_GLOBAL))
        if (H5_UNLIKELY(H5TS_once(&H5TS_first_init_s, H5TS_ONCE_INIT_FUNC) < 0))
            HGOTO_DONE(FAIL);

    
    H5TS_atomic_fetch_add_uint(&H5TS_api_info_p.attempt_lock_count, 1);

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&H5TS_api_info_p.api_mutex) < 0))
        HGOTO_DONE(FAIL);

    
    H5TS_api_info_p.lock_count++;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 
#else
#ifdef H5_HAVE_CONCURRENCY
herr_t
H5TS_api_lock(unsigned *dlftt)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(!H5_INIT_GLOBAL))
        if (H5_UNLIKELY(H5TS_once(&H5TS_first_init_s, H5TS_ONCE_INIT_FUNC) < 0))
            HGOTO_DONE(FAIL);

    
    H5TS_atomic_fetch_add_uint(&H5TS_api_info_p.attempt_lock_count, 1);

    
    if (H5_UNLIKELY(H5TS__get_dlftt(dlftt) < 0))
        HGOTO_DONE(FAIL);

    
    if (0 == *dlftt)
        
        if (H5_UNLIKELY(H5TS_rwlock_wrlock(&H5TS_api_info_p.api_lock) < 0))
            HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 
#else
#error "Unknown multithreading mode"
#endif
#endif

herr_t
H5TS__api_mutex_release(unsigned *lock_count)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

#ifdef H5_HAVE_THREADSAFE
    
    *lock_count = H5TS_api_info_p.lock_count;

    
    H5TS_api_info_p.lock_count = 0;

    
    for (unsigned u = 0; u < *lock_count; u++)
        if (H5_UNLIKELY(H5TS_mutex_unlock(&H5TS_api_info_p.api_mutex) < 0))
            HGOTO_DONE(FAIL);
#else 
    
    if (H5_UNLIKELY(H5TS__get_dlftt(lock_count) < 0))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS__set_dlftt(0) < 0))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_rwlock_wrunlock(&H5TS_api_info_p.api_lock) < 0))
        HGOTO_DONE(FAIL);
#endif

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_api_unlock(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

#ifdef H5_HAVE_THREADSAFE
    
    H5TS_api_info_p.lock_count--;

    
    if (H5_UNLIKELY(H5TS_mutex_unlock(&H5TS_api_info_p.api_mutex) < 0))
        HGOTO_DONE(FAIL);
#else 
    
    if (H5_UNLIKELY(H5TS_rwlock_wrunlock(&H5TS_api_info_p.api_lock) < 0))
        HGOTO_DONE(FAIL);
#endif

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS__tinfo_init(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(H5TS_mutex_init(&H5TS_tinfo_mtx_s, H5TS_MUTEX_TYPE_PLAIN)) < 0)
        ret_value = FAIL;

        
#ifdef H5_HAVE_WIN_THREADS
    if (H5_UNLIKELY(H5TS_key_create(&H5TS_thrd_info_key_g, NULL) < 0))
        ret_value = FAIL;
#else
    if (H5_UNLIKELY(H5TS_key_create(&H5TS_thrd_info_key_g, H5TS__tinfo_destroy) < 0))
        ret_value = FAIL;
#endif

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

static H5TS_tinfo_node_t *
H5TS__tinfo_create(void)
{
    uint64_t           new_id;
    H5TS_tinfo_node_t *tinfo_node;
    H5TS_tinfo_node_t *ret_value;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    
    
    if (H5_UNLIKELY(H5TS_mutex_lock(&H5TS_tinfo_mtx_s) < 0))
        HGOTO_DONE(NULL);

    
    if (NULL != (tinfo_node = H5TS_tinfo_next_free_s))
        H5TS_tinfo_next_free_s = tinfo_node->next;

    
    new_id = ++H5TS_next_thrd_id_s;

    
    if (H5_UNLIKELY(H5TS_mutex_unlock(&H5TS_tinfo_mtx_s) < 0))
        HGOTO_DONE(NULL);

    
    if (NULL == tinfo_node) {
        if (H5_UNLIKELY(NULL == (tinfo_node = H5MM_malloc(sizeof(*tinfo_node)))))
            HGOTO_DONE(NULL);
        tinfo_node->next = NULL;
    }

    
    memset(tinfo_node, 0, sizeof(*tinfo_node));

    
    tinfo_node->info.id = new_id;                       
    H5E__set_default_auto(&tinfo_node->info.err_stack); 

    
    if (H5_UNLIKELY(H5TS_key_set_value(H5TS_thrd_info_key_g, tinfo_node))) {
        H5TS__tinfo_destroy(tinfo_node);
        HGOTO_DONE(NULL);
    }

    
    ret_value = tinfo_node;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
}

herr_t
H5TS_thread_id(uint64_t *id)
{
    H5TS_tinfo_node_t *tinfo_node;
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == id))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_key_get_value(H5TS_thrd_info_key_g, (void **)&tinfo_node) < 0))
        HGOTO_DONE(FAIL);
    if (NULL == tinfo_node)
        
        if (H5_UNLIKELY(NULL == (tinfo_node = H5TS__tinfo_create())))
            HGOTO_DONE(FAIL);

    
    *id = tinfo_node->info.id;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

struct H5CX_node_t **
H5TS_get_api_ctx_ptr(void)
{
    H5TS_tinfo_node_t   *tinfo_node;
    struct H5CX_node_t **ret_value;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(H5TS_key_get_value(H5TS_thrd_info_key_g, (void **)&tinfo_node) < 0))
        HGOTO_DONE(NULL);
    if (NULL == tinfo_node)
        
        if (H5_UNLIKELY(NULL == (tinfo_node = H5TS__tinfo_create())))
            HGOTO_DONE(NULL);

    
    ret_value = &tinfo_node->info.api_ctx_node_ptr;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

H5E_stack_t *
H5TS_get_err_stack(void)
{
    H5TS_tinfo_node_t *tinfo_node;
    H5E_stack_t       *ret_value;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(H5TS_key_get_value(H5TS_thrd_info_key_g, (void **)&tinfo_node) < 0))
        HGOTO_DONE(NULL);
    if (NULL == tinfo_node)
        
        if (H5_UNLIKELY(NULL == (tinfo_node = H5TS__tinfo_create())))
            HGOTO_DONE(NULL);

    
    ret_value = &tinfo_node->info.err_stack;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

#ifdef H5_HAVE_CONCURRENCY

static herr_t
H5TS__get_dlftt(unsigned *dlftt)
{
    H5TS_tinfo_node_t *tinfo_node;
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(H5TS_key_get_value(H5TS_thrd_info_key_g, (void **)&tinfo_node) < 0))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(NULL == tinfo_node))
        
        if (H5_UNLIKELY(NULL == (tinfo_node = H5TS__tinfo_create())))
            HGOTO_DONE(FAIL);

    
    *dlftt = tinfo_node->info.dlftt;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

static herr_t
H5TS__set_dlftt(unsigned dlftt)
{
    H5TS_tinfo_node_t *tinfo_node;
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(H5TS_key_get_value(H5TS_thrd_info_key_g, (void **)&tinfo_node) < 0))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(NULL == tinfo_node))
        
        if (H5_UNLIKELY(NULL == (tinfo_node = H5TS__tinfo_create())))
            HGOTO_DONE(FAIL);

    
    tinfo_node->info.dlftt = dlftt;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

static herr_t
H5TS__inc_dlftt(void)
{
    H5TS_tinfo_node_t *tinfo_node;
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(H5TS_key_get_value(H5TS_thrd_info_key_g, (void **)&tinfo_node) < 0))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(NULL == tinfo_node))
        
        if (H5_UNLIKELY(NULL == (tinfo_node = H5TS__tinfo_create())))
            HGOTO_DONE(FAIL);

    
    tinfo_node->info.dlftt++;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

static herr_t
H5TS__dec_dlftt(void)
{
    H5TS_tinfo_node_t *tinfo_node;
    herr_t             ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(H5TS_key_get_value(H5TS_thrd_info_key_g, (void **)&tinfo_node) < 0))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(NULL == tinfo_node))
        
        if (H5_UNLIKELY(NULL == (tinfo_node = H5TS__tinfo_create())))
            HGOTO_DONE(FAIL);

    
    tinfo_node->info.dlftt--;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 
#endif 

void
H5TS__tinfo_destroy(void *_tinfo_node)
{
    H5TS_tinfo_node_t *tinfo_node = _tinfo_node;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    if (tinfo_node) {
        H5TS_mutex_lock(&H5TS_tinfo_mtx_s);

        
        tinfo_node->next       = H5TS_tinfo_next_free_s;
        H5TS_tinfo_next_free_s = tinfo_node;

        
        H5E__destroy_stack(&tinfo_node->info.err_stack);

        H5TS_mutex_unlock(&H5TS_tinfo_mtx_s);
    }

    FUNC_LEAVE_NOAPI_VOID_NAMECHECK_ONLY
}

int
H5TS_top_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    H5TS__tinfo_term();

    FUNC_LEAVE_NOAPI(n)
}

herr_t
H5TS__tinfo_term(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(H5TS_mutex_lock(&H5TS_tinfo_mtx_s) < 0))
        HGOTO_DONE(FAIL);
    while (H5TS_tinfo_next_free_s) {
        H5TS_tinfo_node_t *next = H5TS_tinfo_next_free_s->next;
        H5MM_free(H5TS_tinfo_next_free_s);
        H5TS_tinfo_next_free_s = next;
    }
    if (H5_UNLIKELY(H5TS_mutex_unlock(&H5TS_tinfo_mtx_s) < 0))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_mutex_destroy(&H5TS_tinfo_mtx_s) < 0))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(H5TS_key_delete(H5TS_thrd_info_key_g) < 0))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

#endif 
