% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/as_ieegio_transform.R
\name{transform_orientation}
\alias{transform_orientation}
\title{Create transform between coordinate orientations}
\usage{
transform_orientation(
  space_from,
  orientation_from,
  orientation_to,
  interpretation = c("active", "passive"),
  image = NULL
)
}
\arguments{
\item{space_from}{either an \code{ieegio_space} object or a character string
identifying the source space. If provided, \code{orientation_from} must
be omitted (orientation is extracted from the space object).}

\item{orientation_from}{character string specifying the source orientation
(e.g., \code{"RAS"}, \code{"LPS"}, \code{"FSL"}). Only used if \code{space_from} is
missing. Must be one of the 48 valid orientation codes plus \code{"FSL"}.}

\item{orientation_to}{character string specifying the target orientation.
Must be one of the 48 valid orientation codes plus \code{"FSL"}.}

\item{interpretation}{character string specifying transform interpretation:
\itemize{
\item \code{"active"} (default): Point transform - transforms point
coordinates from one orientation to another. Use this when you have
coordinates in the source orientation and want to convert them.
\item \code{"passive"}: Axis transform - transforms the coordinate
frame/basis vectors. This is the transpose of the active transform.
Use this when transforming reference frames or basis vectors.
}}

\item{image}{optional image for FSL coordinate conversion. Required when
either \code{orientation_from} or \code{orientation_to} is \code{"FSL"}
(but not both). Can be a file path or an \code{ieegio_volume} object.}
}
\value{
An \code{ieegio_transforms} object containing a 4x4 affine
transformation matrix
}
\description{
Generates an affine transformation to convert coordinates or coordinate
frames between different anatomical orientation conventions (e.g., \code{RAS} to \code{LPS}).
Supports all 48 possible 3D orientations including axis permutations, plus
FSL scaled-voxel coordinates (which require an image for conversion).
}
\details{
The function creates orthogonal transformations (rotations and reflections)
to convert between different anatomical coordinate conventions. For active
transforms, the matrix can be directly applied to homogeneous point coordinates.
For passive transforms, the matrix transforms coordinate axes/frames instead.

Common orientation codes (first 8):
\itemize{
\item \code{RAS}, \code{LAS}, \code{LPS}, \code{RPS}, \code{LPI}, \code{RPI}, \code{LAI}, \code{RAI} (standard axis order)
}

Extended orientations (40 more) include axis permutations like:
\itemize{
\item \code{PIR}, \code{AIL}, \code{SAR}, etc. (permuted axes)
}

\strong{FSL Coordinates:}
When \code{"FSL"} orientation is involved, an image is required for conversion
because FSL coordinates are image-dependent (scaled voxels with possible X-flip).
Three scenarios are supported:
\itemize{
\item \code{FSL -> FSL}: Identity transform (no image needed)
\item \code{FSL -> RAS/other}: Uses \code{vox2ras \%*\% fsl2vox} from image
\item \code{RAS/other -> FSL}: Uses \code{vox2fsl \%*\% ras2vox} from image
}

The relationship between active and passive interpretations:
\code{passive_matrix = t(active_matrix)} for orthogonal transforms.
}
\examples{
# Active transform: convert point coordinates from RAS to LPS
trans <- transform_orientation(orientation_from = "RAS",
                               orientation_to = "LPS",
                               interpretation = "active")
trans$data[[1]]  # diag(-1, -1, 1, 1)

# Apply to a point
point_ras <- c(10, 20, 30, 1)
point_lps <- trans$data[[1]] \%*\% point_ras

# Using a space object
space <- new_space("scanner", orientation = "RAS")
trans <- transform_orientation(space_from = space,
                               orientation_to = "LPS")

# Passive transform: transform coordinate axes
trans_passive <- transform_orientation(orientation_from = "RAS",
                                      orientation_to = "LPS",
                                      interpretation = "passive")

# With axis permutation
trans <- transform_orientation(orientation_from = "RAS",
                               orientation_to = "PIR")

\dontrun{
# FSL to RAS conversion (requires image)
trans_fsl2ras <- transform_orientation(orientation_from = "FSL",
                                       orientation_to = "RAS",
                                       image = "brain.nii.gz")
}

}
