% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimizers-schedules.R
\name{learning_rate_schedule_polynomial_decay}
\alias{learning_rate_schedule_polynomial_decay}
\title{A \code{LearningRateSchedule} that uses a polynomial decay schedule.}
\usage{
learning_rate_schedule_polynomial_decay(
  initial_learning_rate,
  decay_steps,
  end_learning_rate = 0.0001,
  power = 1,
  cycle = FALSE,
  name = "PolynomialDecay"
)
}
\arguments{
\item{initial_learning_rate}{A float. The initial learning rate.}

\item{decay_steps}{A integer. Must be positive. See the decay
computation above.}

\item{end_learning_rate}{A float. The minimal end learning rate.}

\item{power}{A float. The power of the polynomial. Defaults to
\code{1.0}.}

\item{cycle}{A boolean, whether it should cycle beyond decay_steps.}

\item{name}{String.  Optional name of the operation. Defaults to
\code{"PolynomialDecay"}.}
}
\value{
A 1-arg callable learning rate schedule that takes the current optimizer
step and outputs the decayed learning rate, a scalar tensor of the
same type as \code{initial_learning_rate}.
}
\description{
It is commonly observed that a monotonically decreasing learning rate, whose
degree of change is carefully chosen, results in a better performing model.
This schedule applies a polynomial decay function to an optimizer step,
given a provided \code{initial_learning_rate}, to reach an \code{end_learning_rate}
in the given \code{decay_steps}.

It requires a \code{step} value to compute the decayed learning rate. You
can just pass a backend variable that you increment at each training
step.

The schedule is a 1-arg callable that produces a decayed learning rate
when passed the current optimizer step. This can be useful for changing the
learning rate value across different invocations of optimizer functions.
It is computed as:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{decayed_learning_rate <- function(step) \{
  step = min(step, decay_steps)
  ((initial_learning_rate - end_learning_rate) *
    (1 - step / decay_steps) ^ (power)) +
    end_learning_rate
\}
}\if{html}{\out{</div>}}

If \code{cycle} is TRUE then a multiple of \code{decay_steps} is used, the first one
that is bigger than \code{step}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{decayed_learning_rate <- function(step) \{
  decay_steps = decay_steps * ceil(step / decay_steps)
  ((initial_learning_rate - end_learning_rate) *
      (1 - step / decay_steps) ^ (power)) +
    end_learning_rate
\}
}\if{html}{\out{</div>}}

You can pass this schedule directly into a \code{Optimizer}
as the learning rate.
}
\section{Examples}{
Fit a model while decaying from 0.1 to 0.01 in 10000 steps using
sqrt (i.e. power=0.5):

\if{html}{\out{<div class="sourceCode r">}}\preformatted{...
starter_learning_rate <- 0.1
end_learning_rate <- 0.01
decay_steps <- 10000
learning_rate_fn <- learning_rate_schedule_polynomial_decay(
    starter_learning_rate,
    decay_steps,
    end_learning_rate,
    power=0.5)

model \%>\% compile(
  optimizer = optimizer_sgd(learning_rate=learning_rate_fn),
  loss = 'sparse_categorical_crossentropy',
  metrics = 'accuracy'
)

model \%>\% fit(data, labels, epochs=5)
}\if{html}{\out{</div>}}

The learning rate schedule is also serializable and deserializable using
\code{keras$optimizers$schedules$serialize} and
\code{keras$optimizers$schedules$deserialize}.
}

\seealso{
\itemize{
\item \url{https://keras.io/api/optimizers/learning_rate_schedules/polynomial_decay#polynomialdecay-class}
}

Other optimizer learning rate schedules: \cr
\code{\link{LearningRateSchedule}()} \cr
\code{\link{learning_rate_schedule_cosine_decay}()} \cr
\code{\link{learning_rate_schedule_cosine_decay_restarts}()} \cr
\code{\link{learning_rate_schedule_exponential_decay}()} \cr
\code{\link{learning_rate_schedule_inverse_time_decay}()} \cr
\code{\link{learning_rate_schedule_piecewise_constant_decay}()} \cr
}
\concept{optimizer learning rate schedules}
