#######################################################################
# lfl: Linguistic Fuzzy Logic
# Copyright (C) 2025 Michal Burda
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#######################################################################


#' Return vector of values from given interval
#'
#' Returns an ordered vector of values from given interval, of given size,
#' generated by equal steps.
#'
#' Returns a vector of values from \code{from} to \code{to} (inclusive), with
#' equal difference between two consecutive values, with total length \code{n}.
#' Function is useful e.g., together with the \code{\link{pbld}} or
#' \code{\link{defuzz}} functions (for the \code{values} argument; see also
#' \code{\link{lcut}} or \code{\link{fcut}}) or \code{\link{defuzz}}).
#'
#' @param from The lower bound of the interval.
#' @param to The upper bound of the interval.
#' @param n The length of the vector to be produced.
#' @return A vector of numbers in the given interval and size.
#' @author Michal Burda
#' @seealso \code{\link{pbld}}, \code{\link{defuzz}}, \code{\link{fcut}},
#' \code{\link{lcut}}
#' @keywords models robust
#' @examples
#' \dontrun{
#'     slices(1, 5, 10) # 1, 1.5, 2, 2.5, 3, 3.5 4, 4.5, 5
#' }
#' # is the same as
#' seq(1, 5, length.out=10)
#'
#'
#' @export slices
slices <- function(from, to, n) {
    .Deprecated('base::seq')
    if (!is.numeric(from) && length(from) != 1) {
        stop("'from' must be a numeric value")
    }
    if (!is.numeric(to) && length(to) != 1) {
        stop("'to' must be a numeric value")
    }
    if (!is.numeric(n) && length(n) != 1) {
        stop("'n' must be a numeric value")
    }
    if (n < 2) {
        stop("'n' must be at least 2")
    }
    res <- 0:(n-1) / (n-1) * (to - from) + from
    return(res)
}
