% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stamp.R, R/stomp.R, R/scrimp.R, R/mpx.R
\name{stamp}
\alias{stamp}
\alias{stomp}
\alias{scrimp}
\alias{mpx}
\title{Matrix Profile Computation}
\usage{
stamp(
  data,
  window_size,
  query = NULL,
  exclusion_zone = 0.5,
  s_size = 1,
  n_workers = 1,
  progress = TRUE
)

stomp(
  data,
  window_size,
  query = NULL,
  exclusion_zone = 0.5,
  n_workers = 1,
  progress = TRUE,
  left_right_profile = FALSE
)

scrimp(
  data,
  window_size,
  query = NULL,
  exclusion_zone = 0.5,
  s_size = 1,
  pre_scrimp = 0.25,
  n_workers = 1,
  progress = TRUE
)

mpx(
  data,
  window_size,
  query = NULL,
  exclusion_zone = 0.5,
  s_size = 1,
  idxs = TRUE,
  distance = c("euclidean", "pearson"),
  n_workers = 1,
  progress = TRUE
)
}
\arguments{
\item{data}{Required. Any 1-dimension series of numbers (\code{matrix}, \code{vector}, \code{ts} etc.) (See details).}

\item{window_size}{Required. An integer defining the rolling window size.}

\item{query}{(not yet on \code{scrimp()}) Optional. Another 1-dimension series of numbers for an AB-join similarity.
Default is \code{NULL} (See details).}

\item{exclusion_zone}{A numeric. Defines the size of the area around the rolling window that will be ignored to avoid
trivial matches. Default is \code{0.5}, i.e., half of the \code{window_size}.}

\item{s_size}{A numeric. Used on anytime algorithms (stamp, scrimp, mpx) if only part of the computation is needed.
Default is \code{1.0} (means 100\%).}

\item{n_workers}{An integer. The number of threads using for computing. Defaults to \code{1}.}

\item{progress}{A logical. If \code{TRUE} (the default) will show a progress bar. Useful for long computations. (See
details)}

\item{left_right_profile}{(\code{stomp()} only) A boolean. If \code{TRUE}, the function will return the left and right profiles.}

\item{pre_scrimp}{A numeric. If not zero, pre_scrimp is computed, using a fraction of the data. Default is \code{0.25}.
This parameter is ignored when using multithread or AB-join.}

\item{idxs}{(\code{mpx()} only) A logical. Specifies if the computation will return the Profile Index or not. Defaults to
\code{TRUE}.}

\item{distance}{(\code{mpx()} only) A string. Currently accepts \code{euclidean} and \code{pearson}. Defaults to \code{euclidean}.}
}
\value{
Returns a \code{list} with the \code{matrix_profile}, \code{profile_index} (if \code{idxs} is \code{TRUE} in \code{mpx()}), and some
information about the settings used to build it, like \code{ez} and \code{partial} when the algorithm is finished early.
}
\description{
STAMP Computes the best so far Matrix Profile and Profile Index for Univariate Time Series.

STOMP is a faster implementation with the caveat that is not anytime as STAMP or SCRIMP.

SCRIMP is a faster implementation, like STOMP, but has the ability to return anytime results as STAMP.

MPX is by far the fastest implementation with the caveat that is not anytime as STAMP or SCRIMP.
}
\details{
The Matrix Profile, has the potential to revolutionize time series data mining because of its generality,
versatility, simplicity and scalability. In particular it has implications for time series motif discovery, time
series joins, shapelet discovery (classification), density estimation, semantic segmentation, visualization, rule
discovery, clustering etc.

\code{progress}, it is really recommended to use it as feedback for long computations. It indeed adds some
(neglectable) overhead, but the benefit of knowing that your computer is still computing is much bigger than the
seconds you may lose in the final benchmark. About \code{n_workers}, for Windows systems, this package uses TBB for
multithreading, and Linux and macOS, use TinyThread++. This may or not raise some issues in the future, so we must be
aware of slower processing due to different mutexes implementations or even unexpected crashes. The Windows version
is usually more reliable. The \code{data} and \code{query} parameters will be internally converted to a single vector using
\code{as.numeric()}, thus, bear in mind that a multidimensional matrix may not work as you expect, but most 1-dimensional
data types will work normally. If \code{query} is provided, expect the same pre-procesment done for \code{data}; in addition,
\code{exclusion_zone} will be ignored and set to \code{0}. Both \code{data} and \code{query} doesn't need to have the same size and they
can be interchanged if both are provided. The difference will be in the returning object. AB-Join returns the Matrix
Profile 'A' and 'B' i.e., the distance between a rolling window from query to data and from data to query.

\subsection{stamp}{

The anytime STAMP computes the Matrix Profile and Profile Index in such manner that it can be stopped before its
complete calculation and return the best so far results allowing ultra-fast approximate solutions.
}

\subsection{stomp}{

The STOMP uses a faster implementation to compute the Matrix Profile and Profile Index. It can be stopped earlier by
the user, but the result is not considered anytime, just incomplete. For a anytime algorithm, use \code{stamp()} or
\code{scrimp()}.
}

\subsection{scrimp}{

The SCRIMP algorithm was the anytime solution for stomp. It is as fast as stomp but allows the user to cancel the
computation and get an approximation of the final result. This implementation uses the SCRIMP++ code. This means
that, at first, it will compute the pre-scrimp (a very fast and good approximation), and continue improving with
scrimp. The exception is if you use multithreading, that skips the pre-scrimp stage.
}

\subsection{mpx}{

This algorithm was developed apart from the main Matrix Profile branch that relies on Fast Fourier Transform (FFT) at
least in one part of the process. This algorithm doesn't use FFT at all and is several times faster. It also relies
on Ogita's work for better precision computing mean and standard deviation (part of the process).
}
}
\section{This document}{
Last updated on 2025-12-01 using R version 4.2.3.
}

\examples{
mp <- stamp(motifs_discords_small, 50)
mp <- stomp(motifs_discords_small, 50)
mp <- scrimp(motifs_discords_small, 50)
mp <- mpx(motifs_discords_small, 50)
}
\references{
\itemize{
\item Yeh CCM, Zhu Y, Ulanova L, Begum N, Ding Y, Dau HA, et al. Matrix profile I: All pairs similarity joins
for time series: A unifying view that includes motifs, discords and shapelets. Proc - IEEE Int Conf Data Mining,
ICDM. 2017;1317-22.
}

\itemize{
\item Zhu Y, Imamura m, Nikovski D, Keogh E. Matrix Profile VII: Time Series Chains: A New Primitive for Time
Series Data Mining. Knowl Inf Syst. 2018 Jun 2;1-27.
}

\itemize{
\item Zhu Y, Zimmerman Z, Senobari NS, Yeh CM, Funning G. Matrix Profile II : Exploiting a Novel Algorithm
and GPUs to Break the One Hundred Million Barrier for Time Series Motifs and Joins. Icdm. 2016 Jan 22;54(1):739-48.
}

Website: \url{http://www.cs.ucr.edu/~eamonn/MatrixProfile.html}
}
\seealso{
\code{mass()} for the underlying algorithm that finds best match of a query.

\code{mpxab()} for the forward and reverse join-similarity.
}
\concept{matrix profile computations}
