McCullagh_derivative_test_helper <- function(n, x, beta) {
  s <- nrow(n)
  c <- ncol(n)
  gamma <- McCullagh_compute_gamma(x, beta, s, c)
  phi <- matrix(0.0, nrow=s, ncol=(c - 1))
  for (i in 1:s) {
    for (j in 1:(c - 1)) {
      phi[i, j] <- McCullagh_compute_phi(gamma[i, ], j)
    }
  }
  McCullagh_compute_log_l(n, phi)
}


test_that("McCullagh derivative log l wrt beta is correct", {
  n <- vision_data
  s <- nrow(vision_data)
  c <- ncol(vision_data)
  v <- 1
  x <- McCullagh_initialize_x(s, c, v)
  beta <- McCullagh_initialize_beta(n, c, v)
  delta <- 1.0e-7
  epsilon <- 1.0e-5
  s <- nrow(n)
  c <- ncol(n)
  f <- McCullagh_derivative_test_helper(n, x, beta)
  gamma <- McCullagh_compute_gamma(x, beta, s, c)
  g <- McCullagh_derivative_log_l_wrt_beta(n, x, gamma)

  for (k in 1:length(beta)) {
    beta[k] <- beta[k] - delta
    f0 <- McCullagh_derivative_test_helper(n, x, beta)

    beta[k] <- beta[k] + 2.0 * delta
    f1 <- McCullagh_derivative_test_helper(n, x, beta)

    beta[k] <- beta[k] - delta

    der <- ((f - f0) / delta + (f1 - f) / delta) / 2.0
    if (epsilon < abs(g[k])) {
      expect_true(abs((der - g[k]) / g[k]) <= epsilon,
                  info=paste(k, der, g[k]))
    } else {
    expect_true(abs(der - g[k]) <= epsilon,
                info=paste(k, der, g[k]))
    }
  }
}
)


test_that("McCullagh proportional logits example works", {
  R <- McCullagh_compute_cumulative_sums(tonsils)
  cumulative_row_1 <- R[1,]
  result1 <- McCullagh_logits(cumulative_row_1, use_half=TRUE)
  expect_true(abs(result1[1] - (-1.009)) <= 0.0005)
  expect_true(abs(result1[2] - 0.683) <= 0.0005)

  cumulative_row_2 <- R[2,]
  result2 <- McCullagh_logits(cumulative_row_2)
  expect_true(abs(result2[1] - (-0.511)) <= 0.0005)
  expect_true(abs(result2[2] - 1.367) <= 0.0005)

  diff1 <- result1[1] - result2[1]
  diff2 <- result1[2] - result2[2]
  expect_true(abs(diff1 - (-0.498)) <= 0.0005)
  expect_true(abs(diff2 - (-0.684)) <= 0.0005)

  w <- McCullagh_compute_regression_weights(tonsils)$w
  expect_true(abs(w[1] - 0.638) < 0.0005)
  expect_true(abs(w[2] - 0.362) < 0.0005)
  lambda <- McCullagh_compute_lambda(tonsils)

  z <- McCullagh_compute_z(lambda, w)
  delta <- z[1] - z[2]
  expect_true(abs(delta - (-0.565)) <= 0.0005)

  clay <- Clayton_marginal_location(tonsils[1,], tonsils[2,])
  N <- sum(tonsils)
  pi <- colSums(tonsils) / N
  R <- McCullagh_compute_cumulative_sums(tonsils)
  gamma <- colSums(R) / N

  se <- 0.0
  for (j in 1:(ncol(R) - 1)) {
    se <- se + (gamma[j] * (1.0 - gamma[j]) * (pi[j] + pi[j + 1]))
  }
  n <- rowSums(tonsils)
  se <- se * n[1] * n[2] / N

  expect_true(abs(sqrt(1.0 / se) - 0.225) <= 0.0005)
}
)


test_that("McCullagh logits for proportional hazards example works", {

  expected_logits <- matrix(c(-2.67, -1.76, -1.05, -0.02, 0.62, 1.25,
                              -3.10, -2.16, -1.52, -0.79, -0.34, 0.32), nrow = 2, byrow = TRUE)
  R <- McCullagh_compute_cumulative_sums(family_income)
  for (i in 1:nrow(family_income)) {
    actual_logits <- McCullagh_logits(R[i, ], use_half=FALSE)
    for (j in 1:length(actual_logits)) {
      expect_true(abs(expected_logits[i, j] - actual_logits[j]) <= 0.006,
                  info=paste(expected_logits[i, j], actual_logits[j]))
    }
  }

  expected_log_logs <- matrix(c(-2.70, -1.84, -1.20, -0.38, 0.05, 0.41,
                                -3.12, -2.22, -1.62, -0.98, -0.62, -0.14), nrow=2, byrow=TRUE)
  actual_log_logs <- McCullagh_proportional_hazards(family_income)
  for (i in 1:nrow(expected_log_logs)) {
    for (j in 1:ncol(expected_log_logs)) {
      expect_true(abs(expected_log_logs[i, j] - actual_log_logs[i, j]) <= 0.006)
    }
  }
}
)


test_that("McCullagh dreams data example works", {
  R <- colSums(dreams)
  weights <- McCullagh_compute_regression_weights(dreams)
  w <- weights$w
  sum_w <- weights$sum_w / sum(dreams)^3
  expected_sum_w <- 0.297
  expect_true(abs(expected_sum_w - sum_w) <= 0.0005, info=paste(expected_sum_w, sum_w))

  lambda <- McCullagh_compute_lambda(dreams, use_half=TRUE)
  actual_z <- McCullagh_compute_z(lambda, w)
  expected_z <- c(0.2045, 0.5119, -0.3964, -0.3741, -1.4181)
  for (j in length(expected_z)) {
    expect_true(abs(expected_z[j] - actual_z[j]) <= 0.00005)
  }

  expected_vars <- c(0.1603, 0.0687, 0.0673, 0.0571, 0.0765)
  ns <- rowSums(dreams)
  for (i in 1:nrow(dreams)) {
    actual_var <- 1.0 / (sum_w * ns[i])
    expect_true(abs(expected_vars[i] - actual_var) <= 0.00005)
  }
}
)


test_that("McCullagh derivative log l wrt phi is correct", {
  n <- vision_data
  delta <- 1.0e-6
  epsilon <- 1.0e-5
  N <- sum(n)
  R <- McCullagh_compute_cumulative_sums(n)
  r_bar <- colSums(R)
  n_bar <- rowSums(n)
  z <- R / n_bar
  pi <- n / n_bar
  s <- nrow(n)
  c <- ncol(n)

  gamma <- matrix(0.0, nrow=2, ncol=ncol(n))
  gamma[1, ] <- c(0.33, 0.75, 0.90, 1.0)
  gamma[2, ] <- c(0.10, 0.25, 0.67, 1.0)
  phi <- matrix(0.0, nrow=nrow(gamma), ncol=ncol(gamma) - 1)
  for (i in 1:nrow(gamma)) {
    for (j in 1:ncol(phi)) {
      phi[i, j] <- McCullagh_compute_phi(gamma[i,], j)
    }
  }

  f <- McCullagh_compute_log_l(n, phi)

  for (i in 1:nrow(phi)) {
    for (j in 1:(ncol(phi))) {
      phi[i, j] <- phi[i, j] - delta
      f0 <- McCullagh_compute_log_l(n, phi)

      phi[i, j] <- phi[i, j] + 2.0 * delta
      f1 <- McCullagh_compute_log_l(n, phi)

      phi[i, j] <- phi[i, j] - delta

      der <- ((f - f0) / delta + (f1 - f) / delta) / 2.0
      deriv <- McCullagh_derivative_log_l_wrt_phi(n, phi, i, j)
      expect_true(abs(der - deriv) <= epsilon,
                  info=paste(i, j, der, deriv))
    }
  }
}
)


test_that("McCullagh derivative gamma wrt phi is correct", {
  n <- vision_data
  c <- ncol(n)
  v <- 1
  s <- nrow(n)
  beta <- McCullagh_initialize_beta(n, c, v)
  n_beta <- length(beta)
  x <- McCullagh_initialize_x(s, c, v)
  gamma_star <- McCullagh_compute_gamma(x, beta, s, c)
  delta <- 1.0e-7
  epsilon <- 1.0e-6
  for (i in 1:nrow(gamma_star)) {
    gamma <- gamma_star[i,]
    for (j in 1:(ncol(gamma_star) - 1)) {
      phi <- McCullagh_compute_phi(gamma, j)
      f <- McCullagh_compute_gamma_from_phi(phi, j, gamma)
      expect_true(abs(f - gamma[j]) <= epsilon, info=paste("check round trip gamma", f, gamma[j]))

      phi <- phi - delta
      f0 <- McCullagh_compute_gamma_from_phi(phi, j, gamma)

      phi <- phi + 2.0 * delta
      f1 <- McCullagh_compute_gamma_from_phi(phi, j, gamma)

      phi <- phi - delta

      der <- ((f - f0) / delta + (f1 - f) / delta) / 2.0
      deriv <- McCullagh_derivative_gamma_wrt_phi(gamma, j, phi)
      expect_true(abs(der - deriv) <= epsilon)
    }
  }
}
)


test_that("McCullagh derivative gamma plus 1 wrt phi is correct",  {
  n <- vision_data
  c <- ncol(n)
  v <- 1
  s <- nrow(n)
  beta <- McCullagh_initialize_beta(n, c, v)
  n_beta <- length(beta)
  x <- McCullagh_initialize_x(s, c, v)
  gamma_star <- McCullagh_compute_gamma(x, beta, s, c)
  delta <- 1.0e-7
  epsilon <- 1.0e-6
  for (i in 1:nrow(gamma_star)) {
    gamma <- gamma_star[i, ]
    for (j in 1:(ncol(gamma_star) - 1)) {
      phi <- McCullagh_compute_phi(gamma, j)
      f <- McCullagh_compute_gamma_from_phi(phi, j, gamma)
      expect_true(abs(f - gamma[j]) <= epsilon, info=paste("check round trip gamma", f, gamma[j]))

      f <- McCullagh_compute_gamma_plus_1_from_phi(phi, j, gamma)
      phi <- phi - delta
      f0 <- McCullagh_compute_gamma_plus_1_from_phi(phi, j, gamma)

      phi <- phi + 2.0 * delta
      f1 <- McCullagh_compute_gamma_plus_1_from_phi(phi, j, gamma)

      phi <- phi - epsilon

      der <- ((f - f0) / delta + (f1 - f) / delta) / 2.0
      deriv <- McCullagh_derivative_gamma_plus_1_wrt_phi(gamma, j, phi)
      expect_true(abs(der - deriv) <= epsilon, info=paste(j, der, deriv))
    }
  }
}
)


test_that("McCullagh derivative of phi wrt gamma is correct", {
  delta <- 1.0e-6
  epsilon <- 1.0e-5
  gamma <- c(0.33, 0.75, 1.0)
  for (j in 1:2) {
    f <- McCullagh_compute_phi(gamma, j)

    gamma[j] <- gamma[j] - delta
    f0 <- McCullagh_compute_phi(gamma, j)

    gamma[j] <- gamma[j] + 2.0 * delta
    f1 <- McCullagh_compute_phi(gamma, j)

    gamma[j] <- gamma[j] - delta

    der <- ((f - f0) / delta + (f1 - f) / delta) / 2.0
    deriv <- McCullagh_derivative_phi_wrt_gamma(gamma, j)
    expect_true(abs(der - deriv) <= epsilon, info=paste(j, der, deriv))
  }
}
)


test_that("McCullagh location model correctly fits dumping data", {
  # Note that the Agresti (1984) book uses the inverse of logits used here.
  # Thus for comparisons with the book all signs are switched
  n <- dumping
  s <- nrow(n)
  c <- ncol(n)
  n_regression_parm <- 1
  x <- McCullagh_initialize_x(s, c, n_regression_parm)
  result <- McCullagh_fit_location_regression_model(n, x, max_iter=5, verbose=FALSE)

  beta <- result$beta
  expected_beta <- c (0.320, 2.074, -0.225)
  for (k in 1:length(expected_beta)) {
    expect_true(abs(expected_beta[k] - beta[k]) <= 0.0005, info=paste("k", expected_beta[k], beta[k]))
  }

  se <- result$se
  expect_true(abs(0.088 - se[3]) <= 0.0005, info=paste("se beta[2]", 0.088, sqrt(diag(cov))[2]))

  g_squared <- result$g_squared
  expect_true(abs(4.27 - g_squared) <= 0.005, info=paste("G^2", 4.27, g_squared))
}
)


test_that("McCullagh location model fits tonsil data", {
  n <- tonsils
  s <- nrow(n)
  c <- ncol(n)
  n_regression_parm <- 1
  x <- McCullagh_initialize_x(s, c, n_regression_parm)
  result <- McCullagh_fit_location_regression_model(n, x, max_iter=15, verbose=FALSE)

  beta <- result$beta
  expected_beta <- c (-0.810, 1.061, 0.603)
  for (k in 1:length(expected_beta)) {
    expect_true(abs(expected_beta[k] - beta[k]) <= 0.0005, info=paste("k", expected_beta[k], beta[k]))
  }

  se<- result$se
  expect_true(abs(0.116 - se[1]) <= 0.0005, info=paste("se theta[1]", 0.116, sqrt(diag(cov))[2]))
  expect_true(abs(0.118 - se[2]) <= 0.0005, info=paste("se theta[2]", 0.118, sqrt(diag(cov))[2]))
  expect_true(abs(0.225 - se[3]) <= 0.0005, info=paste("se delta", 0.225, sqrt(diag(cov))[2]))

  g_squared <- result$g_squared
  expect_true(abs(0.302 - g_squared) <= 0.005, info=paste("G^2", 0.302, g_squared))
}
)


test_that("taste data fit location model gives right G^2", {
  x0 <- McCullagh_initialize_x(nrow(taste), ncol(taste), 0)
  x1 <- c(rep(1, 4), rep(0, 12), rep(-1, 4))
  x2 <- c(rep(0, 4), rep(1, 4), rep(0, 8), rep(-1, 4))
  x3 <- c(rep(0, 8), rep(1, 4), rep(0, 4), rep(-1, 4))
  x4 <- c(rep(0, 12), rep(1, 4), rep(-1, 4))
  x <- log_linear_append_column(x0, x1)
  x <- log_linear_append_column(x, x2)
  x <- log_linear_append_column(x, x3)
  x <- log_linear_append_column(x, x4)
  result <- McCullagh_fit_location_regression_model(taste, x)
  expect_true(abs(49.1 - result$g_squared) <= 0.05,
              info=paste(49.1, result$g_squared))
  expect_equal(12, result$df)
}
)


test_that("dreams data model 1 fit location model gives right results", {
  # note sign of beta is flipped
  x0 <- McCullagh_initialize_x(nrow(dreams), ncol(dreams), 0)
  x1 <- c(rep(6, 3), rep(8.5, 3), rep(10.5, 3), rep(12.5, 3), rep(14.5, 3))
  x <- log_linear_append_column(x0, x1)
  result <- McCullagh_fit_location_regression_model(dreams, x)
  expect_true(abs(-0.219 - result$beta[4]) <= 0.0005,
              info=paste(0.219, result$beta[4]))
  expect_true(abs(0.0495 - result$se[4]) <= 0.00005,
              info=paste(0.0495, result$se[4]))
  expect_true(abs(12.42 - result$g_squared) <= 0.005,
              info=paste(12.42, result$g_squared))
  expect_equal(11, result$df)
}
)

test_that("dreams data model 2 fit location model gives right G^2", {
  # note signs of betas are flipped
  x0 <- McCullagh_initialize_x(nrow(dreams), ncol(dreams), 0)
  x1 <- c(rep(1, 3), rep(0, 9), rep(-1, 3))
  x2 <- c(rep(0, 3), rep(1, 3), rep(0, 6), rep(-1, 3))
  x3 <- c(rep(0, 6), rep(1, 3), rep(0, 3), rep(-1, 3))
  x4 <- c(rep(0, 9), rep(1, 3), rep(-1, 3))
  x <- log_linear_append_column(x0, x1)
  x <- log_linear_append_column(x, x2)
  x <- log_linear_append_column(x, x3)
  x <- log_linear_append_column(x, x4)
  result <- McCullagh_fit_location_regression_model(dreams, x)
  expect_true(abs(0.615 - result$beta[4]) <= 0.0005)
  expect_true(abs(0.720 - result$beta[5]) <= 0.0005)
  expect_true(abs(-0.077 - result$beta[6]) <= 0.0005)
  expect_true(abs(-0.058 - result$beta[7]) <= 0.0005)
  expect_true(abs(7.15 - result$g_squared) <= 0.005,
              info=paste(7.15, result$g_squared))
  expect_equal(8, result$df)
}
)
