#' @title Non-Compartmental Analysis (NCA) of Plasma Concentration-Time Data (Linear Form)
#' @name non_compartmental
#' @description
#' Performs non-compartmental analysis (NCA) on plasma concentration-time data
#' assuming linear pharmacokinetics. Computes area under the curve (AUC) using trapezoidal rule, estimates
#' terminal elimination rate constant (kel) by linear regression on the log-linear terminal phase,
#' calculates half-life (t1/2), clearance (CL), and volume of distribution (Vd).
#'
#' @param data A data.frame containing plasma concentration-time data.
#' @param time_col Character string indicating the column name for time.
#' @param conc_col Character string indicating the column name for concentration.
#' @param dose Numeric value for the administered dose.
#' @param group_col Optional character string specifying a grouping variable for multiple groups.
#' @param terminal_points Number of last points to use for terminal slope estimation (default = 3).
#' @param plot Logical; if TRUE, plots concentration-time profile and terminal phase regression.
#' @param annotate Logical; if TRUE, annotates plot with PK parameters (only for <= 2 groups).
#'
#' @import scales
#' @import stats
#' @import ggplot2
#' @importFrom scales hue_pal
#' @importFrom stats na.omit lm
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_text geom_abline alpha scale_y_log10
#' labs theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with kel, t1/2, AUC, CL, Vd, and R^2 for each group.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single-subject non-compartmental analysis
#' df <- data.frame(
#'   time = c(0.25, 0.5, 1, 2, 4, 6, 8, 12),
#'   concentration = c(18.6, 16.9, 14.2, 10.8, 6.9, 4.6, 3.1, 1.9)
#' )
#' non_compartmental(
#'   data = df,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,          # mg
#'   terminal_points = 3, # last 3 points (6, 8, 12 h)
#'   plot = TRUE,
#'   annotate = TRUE
#' )
#'
#' # Example II: Two-group comparison (reference vs test formulation)
#' df_groups <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 2),
#'   concentration = c(
#'     17.9, 16.2, 13.7, 10.1, 6.3, 4.1, 2.8,  # Reference
#'     20.4, 18.9, 16.1, 12.4, 8.1, 5.6, 3.9   # Test
#'   ),
#'   formulation = rep(c("Reference", "Test"), each = 7)
#' )
#' non_compartmental(
#'   data = df_groups,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,            # same dose in both groups
#'   group_col = "formulation",
#'   terminal_points = 3,
#'   plot = TRUE,
#'   annotate = TRUE
#' )
#'
#' # Example III: Multiple subjects with extended terminal phase
#' df_subjects <- data.frame(
#'   time = rep(c(0.5, 1, 2, 4, 8, 12, 24), 3),
#'   concentration = c(
#'     15.2, 13.9, 11.4, 7.6, 4.1, 2.6, 1.3,  # Subject 1
#'     14.8, 13.2, 10.9, 7.2, 3.9, 2.4, 1.2,  # Subject 2
#'     16.0, 14.6, 12.0, 8.1, 4.5, 2.9, 1.5   # Subject 3
#'   ),
#'   subject = rep(paste0("S", 1:3), each = 7)
#' )
#' non_compartmental(
#'   data = df_subjects,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 150,
#'   group_col = "subject",
#'   terminal_points = 4,
#'   plot = TRUE,
#'   annotate = FALSE
#' )
#' @references Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
#' 2nd Edition. Marcel Dekker, New York.
#' @references Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
#' Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded.
#' Swedish Pharmaceutical Press, Stockholm.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "conc", "log_conc", "group", "kel", "t_half",
                         "AUC", "CL", "Vd", "R2", "label", "x_pos", "y_pos",
                         "hjust", "vjust", "head", "tail"))

non_compartmental <- function(data,
                              time_col = "time",
                              conc_col = "concentration",
                              dose,
                              group_col = NULL,
                              terminal_points = 3,
                              plot = TRUE,
                              annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE))
    stop("Package 'ggplot2' is required.")
  if (!requireNamespace("scales", quietly = TRUE))
    stop("Package 'scales' is required.")

  # Prepare data
  cols <- c(time_col, conc_col)
  if (!is.null(group_col)) cols <- c(cols, group_col)

  df <- data[, cols, drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "conc")

  df <- df[df$time >= 0 & df$conc > 0, ]
  df$log_conc <- log(df$conc)

  # Group handling
  if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- factor("Experimental")
  }

  # AUC (trapezoidal rule)
  calc_auc <- function(time, conc) {
    sum(diff(time) * (head(conc, -1) + tail(conc, -1)) / 2)
  }

  # Terminal phase fitting
  fit_terminal <- function(d) {
    if (nrow(d) < terminal_points)
      stop("Not enough points to estimate terminal phase slope.")

    terminal_data <- tail(d, terminal_points)
    fit <- stats::lm(log_conc ~ time, data = terminal_data)

    slope <- coef(fit)[2]
    r2 <- summary(fit)$r.squared

    kel <- -slope
    t_half <- log(2) / kel
    auc <- calc_auc(d$time, d$conc)
    CL <- dose / auc
    Vd <- CL / kel

    list(kel = kel, t_half = t_half, auc = auc, CL = CL, Vd = Vd, r2 = r2, fit = fit)
  }

  # Calculate parameters per group
  results <- lapply(split(df, df$group), function(d) {
    term <- fit_terminal(d)
    data.frame(
      group = unique(d$group),
      kel = term$kel,
      t_half = term$t_half,
      AUC = term$auc,
      CL = term$CL,
      Vd = term$Vd,
      R2 = term$r2
    )
  })

  fit_results <- do.call(rbind, results)

  # Plotting
  if (plot) {
    p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = conc, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1) +
      ggplot2::scale_y_log10() +
      ggplot2::labs(
        title = "Non-Compartmental Analysis (Linear Form)",
        subtitle = paste0("Terminal phase last ", terminal_points, " points"),
        x = "Time (hours)",
        y = "Plasma Concentration (log scale)",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid = ggplot2::element_blank()
      )

    # Terminal regression lines
    grp_levels <- levels(df$group)
    cols <- scales::hue_pal()(length(grp_levels))

    for (i in seq_along(grp_levels)) {
      d_grp <- df[df$group == grp_levels[i], ]
      term <- fit_terminal(d_grp)
      fit <- term$fit

      p <- p + ggplot2::geom_abline(
        slope = coef(fit)[2],
        intercept = coef(fit)[1],
        color = ggplot2::alpha(cols[i], 0.7),
        linetype = "dashed",
        linewidth = 1
      )
    }

    # Annotation
    if (annotate && nlevels(df$group) <= 2) {
      ann <- fit_results
      ann$label <- paste0(
        "k_el = ", signif(ann$kel, 3), "\n",
        "t1/2 = ", round(ann$t_half, 2), "\n",
        "AUC = ", round(ann$AUC, 2), "\n",
        "CL = ", round(ann$CL, 2), "\n",
        "Vd = ", round(ann$Vd, 2), "\n",
        "R2 = ", round(ann$R2, 3)
      )

      x_min <- min(df$time); x_max <- max(df$time)
      y_min <- min(df$conc); y_max <- max(df$conc)

      ann$x_pos <- c(x_min + 0.05 * (x_max - x_min),
                     x_max - 0.05 * (x_max - x_min))[seq_len(nrow(ann))]
      ann$y_pos <- c(y_min * 0.5, y_max / 0.5)[seq_len(nrow(ann))]
      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(0, 1)[seq_len(nrow(ann))]

      p <- p + ggplot2::geom_text(
        data = ann,
        ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
        hjust = ann$hjust,
        vjust = ann$vjust,
        size = 4,
        show.legend = FALSE
      )
    }

    print(p)
  }

  list(
    fitted_parameters = fit_results,
    data = df,
    plot = if (plot) p else NULL
  )
}
