#' Write *Marxan* data
#'
#' Write *Marxan* data to disk.
#'
#' @param pu `data.frame` object.
#'
#' @param spec `data.frame` object.
#'
#' @param puvspr `data.frame` object.
#'
#' @param bound `data.frame` object.
#'
#' @param blm `numeric` value.
#'
#' @param asymmetric `numeric` value.
#'
#' @param dir `character` directory to export data.
#'
#' @return A `logical` value indicating success/
#'
#' @noRd
write_marxan_data <- function(pu, spec, puvspr, bound, blm, asymmetric, dir) {
  # assert valid arguments
  assertthat::assert_that(
    is.data.frame(pu),
    is.data.frame(spec),
    is.data.frame(puvspr),
    is.data.frame(bound) || is.null(bound),
    assertthat::is.string(dir)
  )
  # create output directory
  dir.create(dir, showWarnings = FALSE, recursive = TRUE)
  # save data files
  write.table(
    pu, file.path(dir, "pu.dat"),
    sep = ",", row.names = FALSE, append = FALSE
  )
  write.table(
    spec, file.path(dir, "spec.dat"),
    sep = ",", row.names = FALSE, append = FALSE
  )
  write.table(
    puvspr, file.path(dir, "puvspr.dat"),
    sep = ",", row.names = FALSE, append = FALSE
  )
  if (!is.null(bound)) {
    write.table(
      bound, file.path(dir, "bound.dat"),
      sep = ",", row.names = FALSE, append = FALSE
    )
  }
  # save parameter file
  param_text <- paste0("
Input file for Annealing program.

This file generated by Qmarxan
created by Apropos Information Systems Inc.

General Parameters
VERSION 0.1
", ifelse(is.na(blm), "", paste0("BLM ", blm)), "
PROP 5.00000000000000E-0001
RANDSEED -1
BESTSCORE -1.00000000000000E+0000
NUMREPS 10
", ifelse(is.na(asymmetric), "", paste0("ASYMMETRICCONNECTIVITY ", asymmetric)), "

Annealing Parameters
NUMITNS 1000000
STARTTEMP -1.00000000000000E+0000
COOLFAC 0.00000000000000E+0000
NUMTEMP 10000

Cost Threshold
COSTTHRESH 0.00000000000000E+0000
THRESHPEN1 1.40000000000000E-0001
THRESHPEN2 0.00000000000000E+0000

Input File
INPUTDIR ", dir, "
SPECNAME spec.dat
PUNAME pu.dat
PUVSPRNAME puvspr.dat
", ifelse(is.null(bound), "", "BOUNDNAME bound.dat"), "

Save Files
SCENNAME output
SAVERUN 3
SAVEBEST 3
SAVESUMMARY 3
SAVESCEN 2
SAVETARGMET 3
SAVESUMSOLN 3
SAVELOG 1
SAVESNAPSTEPS 0
SAVESNAPCHANGES 0
SAVESNAPFREQUENCY 0
OUTPUTDIR output

Program control.
RUNMODE 1
MISSLEVEL 9.50000000000000E-0001
ITIMPTYPE 1
HEURTYPE 0
CLUMPTYPE 0
VERBOSITY 3

SAVESOLUTIONSMATRIX 3

")
  writeLines(param_text, file.path(dir, "input.dat"))
  # return success
  invisible(TRUE)
}
