% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rem_outliers.R
\name{rem_outliers}
\alias{rem_outliers}
\title{Remove statistical outliers in sky points}
\usage{
rem_outliers(
  sky_points,
  r,
  z,
  a,
  k = 20,
  angular_radius = 20,
  laxity = 2,
  cutoff_side = "both",
  use_window = TRUE,
  trend = NULL
)
}
\arguments{
\item{sky_points}{\code{data.frame} with columns \code{row} and \code{col} (raster
coordinates).}

\item{r}{\link[terra:SpatRaster-class]{terra::SpatRaster}. Image from which \code{sky_points} were
sampled (or any raster with identical dimensions).}

\item{z}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=zenith_image]{zenith_image()}}.}

\item{a}{\link[terra:SpatRaster-class]{terra::SpatRaster} generated with \code{\link[=azimuth_image]{azimuth_image()}}.}

\item{k}{numeric vector of length one. Number of neighbors.}

\item{angular_radius}{numeric vector of length one. The maximum radius for
searching k-nearest neighbors (KNN) in degrees.}

\item{laxity}{numeric vector of length one.}

\item{cutoff_side}{character vector of length one. Options are "both"
(default), "upper" or "lower". Controls which side(s) of the inequality are
evaluated to detect outliers. See Details.}

\item{use_window}{logical of length one. If \code{TRUE} (default), use a
\eqn{3 \times 3} local mean around each point; if \code{FALSE}, use only the
central pixel.}

\item{trend}{numeric vector of length one or \code{NULL}. Zero to three. Specifies
the order of the polynomial surface fitted to the neighbors to account for
spatial trends. Use NULL (default) to skip detrending.}
}
\value{
The retained points represented as a \link{data.frame} with columns \code{row}
and \code{col}, same as \code{sky_points}.
}
\description{
Remove sky points considered outliers relative to their local
neighbors in a user-specified variable.
}
\details{
Based on the Statistical Outlier Removal (SOR) filter from the
\href{https://pointclouds.org/}{PCL library}. Distances are computed on a spherical
surface. The number of neighbors is controlled by \code{k}, and \code{angular_radius}
sets the maximum search radius (deg). If fewer than \code{k} neighbors are found
within that radius, the point is retained due to insufficient evidence for
removal. The decision criterion follows
\insertCite{Leys2013;textual}{rcaiman}:

\eqn{M - laxity \times MAD < x_i < M + laxity \times MAD}

where \eqn{x_i} is the value from \code{r} at sky point \code{i},  \eqn{M} and
\eqn{MAD} are the median and median absolute deviation, respectively,
computed from the the neighbors of \eqn{x_i}, and \eqn{laxity} is the
user-defined threshold.

\code{cutoff_side} controls which side(s) of the inequality are evaluated:
\code{"both"} (default), \code{"left"} (left tail only), or \code{"right"} (right tail
only).
}
\note{
This function assumes that \code{sky_points} and the
\link[terra:SpatRaster-class]{terra::SpatRaster} objects refer to the same image geometry. No checks
are performed.
}
\examples{
\dontrun{
caim <- read_caim()
r <- caim$Blue
z <- zenith_image(ncol(caim), lens())
a <- azimuth_image(z)
m <- !is.na(z)
bin <- binarize_by_region(r, ring_segmentation(z, 30),
                          method = "thr_isodata")
bin <- bin & select_sky_region(z, 0, 80)
g <- sky_grid_segmentation(z, a, 5, first_ring_different = TRUE)
sky_points <- extract_sky_points(r, bin, g,
                                 dist_to_black = 3)
plot(r)
points(sky_points$col, nrow(caim) - sky_points$row, col = 2, pch = 10)

sky_points <- rem_outliers(sky_points, r, z, a,
                                 k = 5,
                                 angular_radius = 20,
                                 laxity = 2,
                                 cutoff_side = "left")
points(sky_points$col, nrow(caim) - sky_points$row, col = 3, pch = 0)
}
}
\references{
\insertAllCited{}
}
