% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/msq.R
\name{msq}
\alias{msq}
\title{Multi-Server Queue Simulation}
\usage{
msq(
  maxArrivals = Inf,
  seed = NA,
  numServers = 2,
  serverSelection = c("LRU", "LFU", "CYC", "RAN", "ORD"),
  interarrivalFcn = NULL,
  serviceFcn = NULL,
  interarrivalType = "M",
  serviceType = "M",
  maxTime = Inf,
  maxDepartures = Inf,
  maxInSystem = Inf,
  maxEventsPerSkyline = 15,
  saveAllStats = FALSE,
  saveInterarrivalTimes = FALSE,
  saveServiceTimes = FALSE,
  saveWaitTimes = FALSE,
  saveSojournTimes = FALSE,
  saveNumInQueue = FALSE,
  saveNumInSystem = FALSE,
  saveServerStatus = FALSE,
  showOutput = TRUE,
  animate = FALSE,
  showQueue = NULL,
  showSkyline = NULL,
  showSkylineSystem = FALSE,
  showSkylineQueue = FALSE,
  showSkylineServer = FALSE,
  showTitle = TRUE,
  showProgress = TRUE,
  plotQueueFcn = defaultPlotMSQ,
  plotSkylineFcn = defaultPlotSkyline,
  jobImage = NA,
  plotDelay = NA,
  respectLayout = FALSE
)
}
\arguments{
\item{maxArrivals}{maximum number of customer arrivals allowed to enter the system}

\item{seed}{initial seed to the random number generator (NA uses current state of
random number generator; NULL seeds using system clock)}

\item{numServers}{Number of servers to simulation (an integer between 1 and 24)}

\item{serverSelection}{Algorithm to use for selecting among idle servers (default is "LRU")}

\item{interarrivalFcn}{Function for generating interarrival times for queue simulation.
Default value (\code{NA}) will result in use of default interarrival
function based on \code{interarrivalType}.  See examples.}

\item{serviceFcn}{Function for generating service times for queue simulation.
Default value (\code{NA}) will result in use of default service function
based on \code{serviceType}.  See examples.}

\item{interarrivalType}{string representation of desired interarrival process.
Options are "M" -- exponential with rate 1;
"G" -- uniform(0,2), having mean 1; and "D" -- deterministic with
constant value 1.  Default is "M".}

\item{serviceType}{string representation of desired service process .
Options are "M" -- exponential with rate 10/9;
"G" -- uniform(0, 1.8), having mean 9/10; and "D" --
deterministic with constant value 9/10.  Default is "M".}

\item{maxTime}{maximum time to simulate}

\item{maxDepartures}{maximum number of customer departures to process}

\item{maxInSystem}{maximum number of customers that the system can hold (server(s) plus queue).
Infinite by default.}

\item{maxEventsPerSkyline}{maximum number of events viewable at a time in the skyline plot.
A large value for this parameter may result in plotting delays.
This parameter does not impact the final plotting, which will show all
end-of-simulation results.}

\item{saveAllStats}{if \code{TRUE}, returns all vectors of statistics (see below)
collected by the simulation}

\item{saveInterarrivalTimes}{if \code{TRUE}, returns a vector of all interarrival times generated}

\item{saveServiceTimes}{if \code{TRUE}, returns a vector of all service times generated}

\item{saveWaitTimes}{if \code{TRUE}, returns a vector of all wait times (in the queue) generated}

\item{saveSojournTimes}{if \code{TRUE}, returns a vector of all sojourn times (time spent in the system)
generated}

\item{saveNumInQueue}{if \code{TRUE}, returns a vector of times and a vector of counts for whenever
the number in the queue changes}

\item{saveNumInSystem}{if \code{TRUE}, returns a vector of times and a vector of counts for whenever
the number in the system changes}

\item{saveServerStatus}{if \code{TRUE}, returns a vector of times and a vector of
server status (0:idle, 1:busy) for whenever the status changes}

\item{showOutput}{if \code{TRUE}, displays summary statistics upon completion}

\item{animate}{If FALSE, no animation will be shown.}

\item{showQueue}{if TRUE, displays a visualization of the queue}

\item{showSkyline}{If \code{NULL} (default), defers to each individual showSkyline...
parameter below; otherwise, supersedes individual showSkyline...
parameter values.
If \code{TRUE}, displays full skyline plot; \code{FALSE} suppresses
skyline plot. Can alternatively be specified using chmod-like octal
component specification: use 1, 2, 4 for system, queue, and server
respectively, summing to indicate desired combination (e.g., 7 for all).
Can also be specified as a binary vector (e.g., c(1,1,1) for all).}

\item{showSkylineSystem}{logical; if \code{TRUE}, includes number in system
as part of skyline plot.  Value for \code{showSkyline}
supersedes this parameter's value.}

\item{showSkylineQueue}{logical; if \code{TRUE}, includes number in queue
as part of skyline plot.  Value for \code{showSkyline}
supersedes this parameter's value.}

\item{showSkylineServer}{logical; if \code{TRUE}, includes number in server
as part of skyline plot.  Value for \code{showSkyline}
supersedes this parameter's value.}

\item{showTitle}{if \code{TRUE}, display title at the top of the main plot}

\item{showProgress}{if TRUE, displays a progress bar on screen during no-animation execution}

\item{plotQueueFcn}{Plotting function to display Queue visualization.
By default, this is provided by \code{defaultPlotSSQ}. Please refer to
the corresponding help for more details about required arguments.}

\item{plotSkylineFcn}{Plotting function to display Skyline visualization.
By default, this is provided by \code{defaultPlotSkyline}.  Please refer
to the corresponding help for more details about required arguments.}

\item{jobImage}{a vector of URLs/local addresses of images to use as jobs. Requires
package \code{'magick'}.}

\item{plotDelay}{a positive numeric value indicating seconds between plots.
A value of -1 enters 'interactive' mode, where the state will pause
for user input at each step.  A value of 0 will display only the final
end-of-simulation plot.}

\item{respectLayout}{If TRUE, plot layout (i.e., par, device, etc.) settings will be respected.
Not recommended except for specialized use.}
}
\value{
The function returns a list containing:
\itemize{
\item the number of arrivals to the system (\code{customerArrivals}),
\item the number of customers processed (\code{customerDepartures}),
\item the ending time of the simulation (\code{simulationEndTime}),
\item average wait time in the queue (\code{avgWait}),
\item average time in the system (\code{avgSojourn}),
\item average number in the system (\code{avgNumInSystem}),
\item average number in the queue (\code{avgNumInQueue}), and
\item server utilization (\code{utilization}).
}
of the queue as computed by the simulation.
When requested via the ``save...'' parameters, the list may also contain:
\itemize{
\item a vector of interarrival times (\code{interarrivalTimes}),
\item a vector of wait times (\code{waitTimes}),
\item a vector of service times (\code{serviceTimes}),
\item a vector of sojourn times (\code{sojournTimes}),
\item two vectors (time and count) noting changes to number in the system
(\code{numInSystemT}, \code{numInSystemN}),
\item two vectors (time and count) noting changes to number in the queue
(\code{numInQueueT}, \code{numInQueueN}), and
\item two vectors (time and status) noting changes to server status
(\code{serverStatusT}, \code{serverStatusN}).
}
}
\description{
A next-event simulation of a single-queue multiple-server
service node, with extensible arrival and service processes.
}
\details{
Implements a next-event implementation of a single-queue multiple-server
queue simulation.

The \code{seed} parameter can take one of three valid
argument types:
\itemize{
\item \code{NA} (default), which will use the current state of the random
number generator without explicitly setting a new seed (see examples);
\item a positive integer, which will be used as the initial seed passed in
an explicit call to \code{\link{set.seed}}; or
\item \code{NULL}, which will be passed in an explicit call to to
\code{\link{set.seed}}, thereby setting the initial seed using the
system clock.
}

The server selection mechanism can be chosen from among five options, with
\code{"LRU"} being the default:
\itemize{
\item \code{"LRU"} (least recently used): from among the currently
available (idle) servers, selects the server who has been idle longest.
\item \code{"LFU"} (least frequently used): from among the currently
available servers, selects the server having the lowest computed
utilization.
\item \code{"CYC"} (cyclic): selects a server in a cyclic manner; i.e,
indexing the servers 1, 2, \eqn{\ldots}, \code{numServers} and incrementing
cyclically, starts from one greater than the index of the most recently
engaged server and selects the first idle server encountered.
\item \code{"RAN"} (random): selects a server at random from among the
currently available servers.
\item \code{"ORD"} (in order): indexing the servers 1, 2, \eqn{\ldots},
\code{numServers}, selects the idle server having the lowest index.
}
}
\examples{
 # process 100 arrivals, R-provided seed (via NULL seed), default 2 servers
 msq(100, NULL)
 # process 100 arrivals, seed 8675309, 3 servers, LFU server selection
 msq(100, 8675309, 3, 'LFU')

 msq(maxArrivals = 100, seed = 8675309)
 msq(maxTime = 100, seed = 8675309)

 ############################################################################
 # example to show use of seed = NA (default) to rely on current state of generator
 output1 <- msq(200, 8675309, showOutput = FALSE, saveAllStats = TRUE)
 output2 <- msq(300,          showOutput = FALSE, saveAllStats = TRUE)
 set.seed(8675309)
 output3 <- msq(200,          showOutput = FALSE, saveAllStats = TRUE)
 output4 <- msq(300,          showOutput = FALSE, saveAllStats = TRUE)
 sum(output1$sojournTimes != output3$sojournTimes) # should be zero
 sum(output2$sojournTimes != output4$sojournTimes) # should be zero

 ############################################################################
 # use same service function for (default) two servers
 myArrFcn <- function() { vexp(1, rate = 1/4, stream = 1) }               # mean is 4
 mySvcFcn <- function() { vgamma(1, shape = 1, rate = 0.3, stream = 2) }  # mean is 3.3
 output <- msq(maxArrivals = 100, interarrivalFcn = myArrFcn,
     serviceFcn = mySvcFcn, saveAllStats = TRUE)
 mean(output$interarrivalTimes)
 mean(output$serviceTimes)

 ############################################################################
 # use different service function for (default) two servers
 myArrFcn  <- function() { vexp(1, rate = 1/4, stream = 1) }                # mean is 4
 mySvcFcn1 <- function() { vgamma(1, shape = 3, scale = 1.1, stream = 2) }  # mean is 3.3
 mySvcFcn2 <- function() { vgamma(1, shape = 3, scale = 1.2, stream = 3) }  # mean is 3.6
 output <- msq(maxArrivals = 100, interarrivalFcn = myArrFcn,
     serviceFcn = list(mySvcFcn1, mySvcFcn2), saveAllStats = TRUE)
 mean(output$interarrivalTimes)
 meanTPS(output$numInQueueT, output$numInQueueN)  # compute time-averaged num in queue
 mean(output$serviceTimesPerServer[[1]])  # compute avg service time for server 1
 mean(output$serviceTimesPerServer[[2]])  # compute avg service time for server 2
 meanTPS(output$serverStatusT[[1]], output$serverStatusN[[1]])  # compute server 1 utilization
 meanTPS(output$serverStatusT[[2]], output$serverStatusN[[2]])  # compute server 2 utilization

 ############################################################################
 # example to show use of (simple) trace data for arrivals and service times,
 # allowing for reuse of trace data times
 smallQueueTrace <- list()
 smallQueueTrace$arrivalTimes <- c(15, 47, 71, 111, 123, 152, 166, 226, 310, 320)
 smallQueueTrace$serviceTimes <- c(43, 36, 34,  30,  38,  40,  31,  29,  36,  30)

 interarrivalTimes <- NULL
 serviceTimes      <- NULL

 getInterarr <- function()
 {
     if (length(interarrivalTimes) == 0) {
           interarrivalTimes <<- c(smallQueueTrace$arrivalTimes[1],
                                  diff(smallQueueTrace$arrivalTimes))
     }
     nextInterarr <- interarrivalTimes[1]
     interarrivalTimes <<- interarrivalTimes[-1]  # remove 1st element globally
     return(nextInterarr)
 }

 getService <- function()
 {
     if (length(serviceTimes) == 0) {
         serviceTimes <<- smallQueueTrace$serviceTimes
     }
     nextService <- serviceTimes[1]
     serviceTimes <<- serviceTimes[-1]  # remove 1st element globally
     return(nextService)
 }

 output <- msq(maxArrivals = 100, numServers = 2, interarrivalFcn = getInterarr,
               serviceFcn = getService, saveAllStats = TRUE)
 mean(output$interarrivalTimes)
 mean(output$serviceTimes)
 mean(output$serviceTimesPerServer[[1]])  # compute avg service time for server 1
 mean(output$serviceTimesPerServer[[2]])  # compute avg service time for server 2

 ############################################################################
 # Testing with visualization

 # Visualizing msq with a set seed, infinite queue capacity, 10 arrivals,
 # and showing queue (default) and skyline for all 3 attributes
 msq(seed = 1234, numServers = 5, maxArrivals = 10, showSkyline = 7, 
     plotDelay = 0.1)

 # Same simulation as above but using default interactive mode
 if (interactive()) {
   msq(seed = 1234, numServers = 5, maxArrivals = 10, showSkyline = 7)
 }

 # Visualizing msq with a set seed, finite queue capacity, 20 arrivals,
 # and showing queue (default) and skyline for all 3 attributes
 msq(seed = 1234, numServers = 5, maxArrivals = 25, showSkyline = 7,
     maxInSystem = 5, plotDelay = 0)

 # Using default distributions to simulate an M/G/2 queue
 msq(seed = 1234, maxDepartures = 10, 
     interarrivalType = "M", serviceType = "G", plotDelay = 0)

}
\seealso{
\code{\link[rstream]{rstream}}, \code{\link{set.seed}},
\code{\link[=runif]{stats::runif}}
}
\author{
Barry Lawson (\email{blawson@bates.edu}), \cr
Larry Leemis (\email{leemis@math.wm.edu}), \cr
Vadim Kudlay (\email{vkudlay@nvidia.com})
}
\concept{queueing}
\keyword{utilities}
