% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/roll.R
\name{roll_sum}
\alias{roll_sum}
\alias{roll_mean}
\alias{roll_geometric_mean}
\alias{roll_harmonic_mean}
\alias{roll_growth_rate}
\title{Fast by-group rolling functions}
\usage{
roll_sum(
  x,
  window = Inf,
  g = NULL,
  partial = TRUE,
  weights = NULL,
  na.rm = TRUE,
  ...
)

roll_mean(
  x,
  window = Inf,
  g = NULL,
  partial = TRUE,
  weights = NULL,
  na.rm = TRUE,
  ...
)

roll_geometric_mean(
  x,
  window = Inf,
  g = NULL,
  partial = TRUE,
  weights = NULL,
  na.rm = TRUE,
  ...
)

roll_harmonic_mean(
  x,
  window = Inf,
  g = NULL,
  partial = TRUE,
  weights = NULL,
  na.rm = TRUE,
  ...
)

roll_growth_rate(
  x,
  window = Inf,
  g = NULL,
  partial = TRUE,
  na.rm = FALSE,
  log = FALSE,
  inf_fill = NULL
)
}
\arguments{
\item{x}{Numeric vector, data frame, or list.}

\item{window}{Rolling window size, default is \code{Inf}.}

\item{g}{Grouping object passed directly to \code{collapse::GRP()}.
This can for example be a vector or data frame.}

\item{partial}{Should calculations be done using partial windows?
Default is \code{TRUE}.}

\item{weights}{Importance weights. Must be the same length as x.
Currently, no normalisation of weights occurs.}

\item{na.rm}{Should missing values be removed for the calculation?
The default is \code{TRUE}.}

\item{...}{Additional arguments passed to \code{data.table::frollmean} and
\code{data.table::frollsum}.}

\item{log}{For \code{roll_growth_rate}:
If \code{TRUE} then growth rates are calculated on the log-scale.}

\item{inf_fill}{For \code{roll_growth_rate}:
Numeric value to replace \code{Inf} values with.
Default behaviour is to keep \code{Inf} values.}
}
\value{
A numeric vector the same length as \code{x} when \code{x} is a vector,
or a list when \code{x} is a \code{data.frame}. \cr
}
\description{
An efficient method for rolling sum, mean and growth rate for many groups.
}
\details{
\code{roll_sum} and \code{roll_mean} support parallel computations when
\code{x} is a data frame of multiple columns. \cr
\code{roll_geometric_mean} and \code{roll_harmonic_mean} are convenience functions that
utilise \code{roll_mean}. \cr
\code{roll_growth_rate} calculates the rate of percentage
change per unit time on a rolling basis.
}
\examples{
library(timeplyr)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 1L)
collapse::set_collapse(nthreads = 1L)
}
x <- 1:10
roll_sum(x) # Simple rolling total
roll_mean(x) # Simple moving average
roll_sum(x, window = 3)
roll_mean(x, window = 3)
roll_sum(x, window = 3, partial = FALSE)
roll_mean(x, window = 3, partial = FALSE)

# Plot of expected value of 'coin toss' over many flips
set.seed(42)
x <- sample(c(1, 0), 10^3, replace = TRUE)
ev <- roll_mean(x)
plot(ev)
abline(h = 0.5, lty = 2)

all.equal(roll_sum(iris$Sepal.Length, g = iris$Species),
          ave(iris$Sepal.Length, iris$Species, FUN = cumsum))
# The below is run using parallel computations where applicable
roll_sum(iris[, 1:4], window = 7, g = iris$Species)
\donttest{
  library(data.table)
  library(bench)
  df <- data.table(g = sample.int(10^4, 10^5, TRUE),
                   x = rnorm(10^5))
  mark(e1 = df[, mean := frollmean(x, n = 7,
                                   align = "right", na.rm = FALSE), by = "g"]$mean,
       e2 = df[, mean := roll_mean(x, window = 7, g = get("g"),
                                   partial = FALSE, na.rm = FALSE)]$mean)
}
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
\seealso{
\link{time_roll_mean}
}
