










use rand_core::{Error, RngCore};
use crate::distributions::uniform::{SampleRange, SampleUniform};
use crate::distributions::{self, Distribution, Standard};
use core::num::Wrapping;
use core::{mem, slice};






/// # Generic usage






















/// # use rand::thread_rng;






/// # let v = foo(&mut thread_rng());

pub trait Rng: RngCore {
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    /// # Arrays and tuples
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[inline]
    fn gen<T>(&mut self) -> T
    where Standard: Distribution<T> {
        Standard.sample(self)
    }

    
    
    
    
    
    
    
    
    /// # Panics
    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    fn gen_range<T, R>(&mut self, range: R) -> T
    where
        T: SampleUniform,
        R: SampleRange<T>
    {
        assert!(!range.is_empty(), "cannot sample empty range");
        range.sample_single(self)
    }

    
    
    /// ### Example
    
    
    
    
    
    
    
    
    
    
    
    fn sample<T, D: Distribution<T>>(&mut self, distr: D) -> T {
        distr.sample(self)
    }

    
    
    
    
    
    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    fn sample_iter<T, D>(self, distr: D) -> distributions::DistIter<D, Self, T>
    where
        D: Distribution<T>,
        Self: Sized,
    {
        distr.sample_iter(self)
    }

    
    
    
    
    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    fn fill<T: Fill + ?Sized>(&mut self, dest: &mut T) {
        dest.try_fill(self).unwrap_or_else(|_| panic!("Rng::fill failed"))
    }

    
    
    
    
    
    
    
    /// # Example
    
    
    /// # use rand::Error;
    
    
    /// # fn try_inner() -> Result<(), Error> {
    
    
    /// # Ok(())
    /// # }
    
    /// # try_inner().unwrap()
    
    
    
    
    fn try_fill<T: Fill + ?Sized>(&mut self, dest: &mut T) -> Result<(), Error> {
        dest.try_fill(self)
    }

    
    
    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    /// # Panics
    
    
    
    
    #[inline]
    fn gen_bool(&mut self, p: f64) -> bool {
        let d = distributions::Bernoulli::new(p).unwrap();
        self.sample(d)
    }

    
    
    
    
    
    
    
    
    
    /// # Panics
    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    #[inline]
    fn gen_ratio(&mut self, numerator: u32, denominator: u32) -> bool {
        let d = distributions::Bernoulli::from_ratio(numerator, denominator).unwrap();
        self.sample(d)
    }
}

impl<R: RngCore + ?Sized> Rng for R {}








pub trait Fill {
    
    fn try_fill<R: Rng + ?Sized>(&mut self, rng: &mut R) -> Result<(), Error>;
}

macro_rules! impl_fill_each {
    () => {};
    ($t:ty) => {
        impl Fill for [$t] {
            fn try_fill<R: Rng + ?Sized>(&mut self, rng: &mut R) -> Result<(), Error> {
                for elt in self.iter_mut() {
                    *elt = rng.gen();
                }
                Ok(())
            }
        }
    };
    ($t:ty, $($tt:ty,)*) => {
        impl_fill_each!($t);
        impl_fill_each!($($tt,)*);
    };
}

impl_fill_each!(bool, char, f32, f64,);

impl Fill for [u8] {
    fn try_fill<R: Rng + ?Sized>(&mut self, rng: &mut R) -> Result<(), Error> {
        rng.try_fill_bytes(self)
    }
}

macro_rules! impl_fill {
    () => {};
    ($t:ty) => {
        impl Fill for [$t] {
            #[inline(never)] // in micro benchmarks, this improves performance
            fn try_fill<R: Rng + ?Sized>(&mut self, rng: &mut R) -> Result<(), Error> {
                if self.len() > 0 {
                    rng.try_fill_bytes(unsafe {
                        slice::from_raw_parts_mut(self.as_mut_ptr()
                            as *mut u8,
                            self.len() * mem::size_of::<$t>()
                        )
                    })?;
                    for x in self {
                        *x = x.to_le();
                    }
                }
                Ok(())
            }
        }

        impl Fill for [Wrapping<$t>] {
            #[inline(never)]
            fn try_fill<R: Rng + ?Sized>(&mut self, rng: &mut R) -> Result<(), Error> {
                if self.len() > 0 {
                    rng.try_fill_bytes(unsafe {
                        slice::from_raw_parts_mut(self.as_mut_ptr()
                            as *mut u8,
                            self.len() * mem::size_of::<$t>()
                        )
                    })?;
                    for x in self {
                    *x = Wrapping(x.0.to_le());
                    }
                }
                Ok(())
            }
        }
    };
    ($t:ty, $($tt:ty,)*) => {
        impl_fill!($t);
        // TODO: this could replace above impl once Rust #32463 is fixed
        
        impl_fill!($($tt,)*);
    }
}

impl_fill!(u16, u32, u64, usize, u128,);
impl_fill!(i8, i16, i32, i64, isize, i128,);

#[cfg_attr(doc_cfg, doc(cfg(feature = "min_const_gen")))]
#[cfg(feature = "min_const_gen")]
impl<T, const N: usize> Fill for [T; N]
where [T]: Fill
{
    fn try_fill<R: Rng + ?Sized>(&mut self, rng: &mut R) -> Result<(), Error> {
        self[..].try_fill(rng)
    }
}

#[cfg(not(feature = "min_const_gen"))]
macro_rules! impl_fill_arrays {
    ($n:expr,) => {};
    ($n:expr, $N:ident) => {
        impl<T> Fill for [T; $n] where [T]: Fill {
            fn try_fill<R: Rng + ?Sized>(&mut self, rng: &mut R) -> Result<(), Error> {
                self[..].try_fill(rng)
            }
        }
    };
    ($n:expr, $N:ident, $($NN:ident,)*) => {
        impl_fill_arrays!($n, $N);
        impl_fill_arrays!($n - 1, $($NN,)*);
    };
    (!div $n:expr,) => {};
    (!div $n:expr, $N:ident, $($NN:ident,)*) => {
        impl_fill_arrays!($n, $N);
        impl_fill_arrays!(!div $n / 2, $($NN,)*);
    };
}
#[cfg(not(feature = "min_const_gen"))]
#[rustfmt::skip]
impl_fill_arrays!(32, N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,N,);
#[cfg(not(feature = "min_const_gen"))]
impl_fill_arrays!(!div 4096, N,N,N,N,N,N,N,);

#[cfg(test)]
mod test {
    use super::*;
    use crate::test::rng;
    use crate::rngs::mock::StepRng;
    #[cfg(feature = "alloc")] use alloc::boxed::Box;

    #[test]
    fn test_fill_bytes_default() {
        let mut r = StepRng::new(0x11_22_33_44_55_66_77_88, 0);

        
        let lengths = [0, 1, 2, 3, 4, 5, 6, 7, 80, 81, 82, 83, 84, 85, 86, 87];
        for &n in lengths.iter() {
            let mut buffer = [0u8; 87];
            let v = &mut buffer[0..n];
            r.fill_bytes(v);

            
            for (i, &byte) in v.iter().enumerate() {
                if byte == 0 {
                    panic!("byte {} of {} is zero", i, n)
                }
            }
        }
    }

    #[test]
    fn test_fill() {
        let x = 9041086907909331047; 
        let mut rng = StepRng::new(x, 0);

        
        let mut array = [0u64; 2];
        rng.fill(&mut array[..]);
        assert_eq!(array, [x, x]);
        assert_eq!(rng.next_u64(), x);

        
        let mut array = [0u32; 2];
        rng.fill(&mut array[..]);
        assert_eq!(array, [x as u32, (x >> 32) as u32]);
        assert_eq!(rng.next_u32(), x as u32);

        
        let mut warray = [Wrapping(0u32); 2];
        rng.fill(&mut warray[..]);
        assert_eq!(array[0], warray[0].0);
        assert_eq!(array[1], warray[1].0);

        
        let mut array = [0f32; 2];
        rng.fill(&mut array);
        let gen: [f32; 2] = rng.gen();
        assert_eq!(array, gen);
    }

    #[test]
    fn test_fill_empty() {
        let mut array = [0u32; 0];
        let mut rng = StepRng::new(0, 1);
        rng.fill(&mut array);
        rng.fill(&mut array[..]);
    }

    #[test]
    fn test_gen_range_int() {
        let mut r = rng(101);
        for _ in 0..1000 {
            let a = r.gen_range(-4711..17);
            assert!((-4711..17).contains(&a));
            let a: i8 = r.gen_range(-3..42);
            assert!((-3..42).contains(&a));
            let a: u16 = r.gen_range(10..99);
            assert!((10..99).contains(&a));
            let a: i32 = r.gen_range(-100..2000);
            assert!((-100..2000).contains(&a));
            let a: u32 = r.gen_range(12..=24);
            assert!((12..=24).contains(&a));

            assert_eq!(r.gen_range(0u32..1), 0u32);
            assert_eq!(r.gen_range(-12i64..-11), -12i64);
            assert_eq!(r.gen_range(3_000_000..3_000_001), 3_000_000);
        }
    }

    #[test]
    fn test_gen_range_float() {
        let mut r = rng(101);
        for _ in 0..1000 {
            let a = r.gen_range(-4.5..1.7);
            assert!((-4.5..1.7).contains(&a));
            let a = r.gen_range(-1.1..=-0.3);
            assert!((-1.1..=-0.3).contains(&a));

            assert_eq!(r.gen_range(0.0f32..=0.0), 0.);
            assert_eq!(r.gen_range(-11.0..=-11.0), -11.);
            assert_eq!(r.gen_range(3_000_000.0..=3_000_000.0), 3_000_000.);
        }
    }

    #[test]
    #[should_panic]
    fn test_gen_range_panic_int() {
        #![allow(clippy::reversed_empty_ranges)]
        let mut r = rng(102);
        r.gen_range(5..-2);
    }

    #[test]
    #[should_panic]
    fn test_gen_range_panic_usize() {
        #![allow(clippy::reversed_empty_ranges)]
        let mut r = rng(103);
        r.gen_range(5..2);
    }

    #[test]
    fn test_gen_bool() {
        #![allow(clippy::bool_assert_comparison)]

        let mut r = rng(105);
        for _ in 0..5 {
            assert_eq!(r.gen_bool(0.0), false);
            assert_eq!(r.gen_bool(1.0), true);
        }
    }

    #[test]
    fn test_rng_trait_object() {
        use crate::distributions::{Distribution, Standard};
        let mut rng = rng(109);
        let mut r = &mut rng as &mut dyn RngCore;
        r.next_u32();
        r.gen::<i32>();
        assert_eq!(r.gen_range(0..1), 0);
        let _c: u8 = Standard.sample(&mut r);
    }

    #[test]
    #[cfg(feature = "alloc")]
    fn test_rng_boxed_trait() {
        use crate::distributions::{Distribution, Standard};
        let rng = rng(110);
        let mut r = Box::new(rng) as Box<dyn RngCore>;
        r.next_u32();
        r.gen::<i32>();
        assert_eq!(r.gen_range(0..1), 0);
        let _c: u8 = Standard.sample(&mut r);
    }

    #[test]
    #[cfg_attr(miri, ignore)] // Miri is too slow
    fn test_gen_ratio_average() {
        const NUM: u32 = 3;
        const DENOM: u32 = 10;
        const N: u32 = 100_000;

        let mut sum: u32 = 0;
        let mut rng = rng(111);
        for _ in 0..N {
            if rng.gen_ratio(NUM, DENOM) {
                sum += 1;
            }
        }
        
        let expected = (NUM * N) / DENOM; 
        assert!(((sum - expected) as i32).abs() < 500);
    }
}
