% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{FluxPoint}
\alias{FluxPoint}
\title{FluxPoint change point detection algorithm}
\usage{
FluxPoint(
  data,
  w,
  tc,
  max_iter1,
  max_iter2,
  ignoreCross = FALSE,
  noeta = FALSE,
  nophi = FALSE,
  needReproduce = FALSE
)
}
\arguments{
\item{data}{Numeric matrix of dimension \eqn{n \times p} containing the
observed multivariate time series.}

\item{w}{Integer specifying the window size used by the detector.}

\item{tc}{Numeric tuning constant used in the detection threshold
\eqn{D = \texttt{tc} \cdot \min(4, \log(e^2 + p)) \cdot \log(n - w)}.}

\item{max_iter1}{Integer specifying the maximum number of iterations for
the first-stage loop, which alternates between diagonal robust parameter
estimation and change point detection.}

\item{max_iter2}{Integer specifying the maximum number of iterations for
the second-stage refinement loop, which incorporates non-diagonal
vector autoregressive updates.}

\item{ignoreCross}{Logical; if \code{TRUE}, the algorithm terminates after
the first stage and treats the components of the time series as independent.}

\item{noeta}{Logical; if \code{TRUE}, forces \eqn{\Sigma_{\boldsymbol{\eta}} = 0}
and performs change point detection without accounting for random walk
fluctuations in the mean.}

\item{nophi}{Logical; if \code{TRUE}, forces \eqn{\Phi = 0} and performs
change point detection without accounting for temporal dependence. This
option should only be used when \code{ignoreCross = TRUE}.}

\item{needReproduce}{Logical; if \code{TRUE}, fixed folds are used in
internal cross-validation steps to improve reproducibility.}
}
\value{
A list containing:
\itemize{
  \item \code{cps}: Sorted indices of the detected change points.
  \item \code{Sig_eta_hat}: Final estimate of \eqn{\Sigma_{\boldsymbol{\eta}}}.
  \item \code{Sig_nu_hat}: Final estimate of \eqn{\Sigma_{\boldsymbol{\nu}}},
    which may be non-diagonal if the second-stage refinement is performed.
  \item \code{Phi_hat}: Final estimate of \eqn{\Phi}, which may be
    non-diagonal if the second-stage refinement is performed.
  \item \code{muhats}: Estimated fluctuating mean sequence.
  \item \code{detectorStats}: Detector statistic evaluated over time.
  \item \code{cps_at}: A list mapping each detected change point to the
    indices of components selected as contributing to that change.
}
}
\description{
Implements the full FluxPoint algorithm for detecting multiple change points
in multivariate time series with non-stationary dynamics and cross-correlations.
The procedure iteratively estimates model parameters and change point
locations, alternating between parameter estimation and detection
steps until convergence.
}
\details{
The algorithm proceeds through the following stages:
\enumerate{
  \item \emph{Stage I (diagonal estimation):} Robust parameter
  estimation is performed to obtain diagonal estimates of
  \eqn{\Sigma_{\boldsymbol{\eta}}}, \eqn{\Sigma_{\boldsymbol{\nu}}}, and
  \eqn{\Phi}. These estimates are used to construct the windowed covariance
  matrix \eqn{\Sigma_{\mathrm{ALL}}^{*}} and its inverse. Change point
  detection is then carried out using the resulting detector statistic.
  The estimation and detection steps are iterated until the detected change
  points stabilize or \code{max_iter1} is reached.

  \item \emph{Stage II (refinement with cross-correlation):} If enabled,
  the fluctuating mean is estimated segmentwise and removed from the data.
  A sparse vector autoregressive model is then fitted to the residuals to
  obtain non-diagonal estimates of \eqn{\Phi} and
  \eqn{\Sigma_{\boldsymbol{\nu}}}. The covariance matrix
  \eqn{\Sigma_{\mathrm{ALL}}^{*}} is recomputed and change point detection
  is rerun. This refinement loop is repeated until convergence or until
  \code{max_iter2} is reached.
}
}
\examples{
## Minimal runnable example (fast)
set.seed(123)
p <- 1
mu0 <- rep(0, p)
deltas <- list(c(3), c(-3))
Sig_eta <- diag(0.01, p)
Sig_nu  <- random_Signu(p, 0)
Phi     <- random_Phi(p, 0)
Sig_e1  <- get_Sig_e1_approx(Sig_nu, Phi)

# Simulate data with two evenly spaced change points
Y <- generate_data(mu0, deltas, Sig_eta, Sig_nu, Phi, Sig_e1,
                   errortype = "n", number_cps = 2, lengthofeachpart = 100)

# Run the algorithm
out <- FluxPoint(Y, w = 20, tc = 1, max_iter1 = 5, max_iter2 = 5)
out$cps
# Visualization
p1 <- plot_FluxPoint(Y, out$muhats, out$cps, titlename = "", xaxis = "Time")
print(p1)

\donttest{
## More realistic example (may take longer)
set.seed(123)
p <- 3
mu0 <- rep(0, p)
deltas <- list(c(3, 0, -3), c(0, -2, 4))
Sig_eta <- diag(0.01, p)
Sig_nu  <- random_Signu(p, 0)
Phi     <- random_Phi(p, p)
Sig_e1  <- get_Sig_e1_approx(Sig_nu, Phi)

Y <- generate_data(mu0, deltas, Sig_eta, Sig_nu, Phi, Sig_e1,
                   errortype = "n", number_cps = 2, lengthofeachpart = 100)

out <- FluxPoint(Y, w = 20, tc = 1, max_iter1 = 5, max_iter2 = 5)
out$cps

# Visualization
p1 <- plot_FluxPoint(Y, out$muhats, out$cps, titlename = "", xaxis = "Time")
print(p1)
}

}
\references{
Tian, Y. and Safikhani, A. (2025).
Multiple change point detection in time series with non-stationary dynamics.
Manuscript under review.
}
