#' @export
#' @title Make a Policy of Sticky Price
#' @aliases makePolicyStickyPrice
#' @description Given a stickiness value and a time window vector, this function returns a policy function that
#' sets the current prices equal to the weighted mean of the market-clearing prices and the current prices during this time window.
#' When the stickiness value is 0, the prices will be set to the market-clearing prices.
#' When the stickiness value is 1, the current prices will keep unchanged.
#' @param stickiness a stickiness value between 0 and 1.
#' @param time.win the time window vector, i.e. a 2-vector specifying the start time and end time of policy implementation.
#' @param tolCond the tolerance condition for computing the market-clearing price vector.
#' @return A policy function, which is often used as an argument of the function sdm2.
#' @note Three major price adjustment methods can be used in the structural dynamic model.
#' The corresponding three kinds of prices are exploratory prices (the default case), market clearing prices, and sticky prices.
#' The exploratory prices are computed based on the prices and sales rates of the previous period.
#' In economic reality, the market clearing prices are unknown, so exploratory prices are more realistic.
#' @note
#' When the stickiness value is positive and the priceAdjustmentVelocity parameter in sdm2 is set to 0
#' (indicating that current prices remain unchanged from the previous period), implementing the sticky-price policy results in
#' current prices that are the weighted average of the market-clearing prices and the prices from the previous period.
#' Typically, this function should be utilized in this manner.
#' @seealso \code{\link{sdm2}}
#' @examples
#' \donttest{
#' InitialEndowments <- {
#'   tmp <- matrix(0, 3, 2)
#'   tmp[1, 1] <- 0.01
#'   tmp[2, 2] <- tmp[3, 2] <- 1
#'   tmp
#' }
#'
#' ge <- gemCanonicalDynamicMacroeconomic_3_2(
#'   priceAdjustmentVelocity = 0,
#'   policy.supply = makePolicySupply(InitialEndowments),
#'   policy.price = makePolicyStickyPrice(stickiness = 0.5),
#'   ts = TRUE,
#'   maxIteration = 1,
#'   numberOfPeriods = 50
#' )
#'
#' par(mfrow = c(1, 2))
#' matplot(ge$ts.z, type = "o", pch = 20)
#' matplot(ge$ts.p, type = "o", pch = 20)
#' }
#'
makePolicyStickyPrice <- function(stickiness = 0.5, time.win = c(1, Inf), tolCond = 1e-6) {
  function(time, state, A) {
    if (time >= time.win[1] && time <= time.win[2]) {
      spot.equilibrium <- sdm2(A,
        B = 0 * state$S, S0Exg = state$S,
        names.commodity = state$names.commodity,
        names.agent = state$names.agent,
        tolCond = tolCond
      )
      p.market.clearing <- spot.equilibrium$p
      state$p <- state$p * stickiness + (1 - stickiness) * p.market.clearing
    }

    state
  }
}
