% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MCNM.R
\name{MCNM}
\alias{MCNM}
\title{Multivariate Contaminated Normal Mixture (MCNM)}
\usage{
MCNM(
  X,
  G,
  criterion = c("BIC", "AIC", "KIC", "KICc", "AIC3", "CAIC", "AICc", "ICL", "AWE", "CLC"),
  max_iter = 20,
  epsilon = 0.01,
  init_method = c("kmedoids", "kmeans", "hierarchical", "mclust", "manual"),
  clusters = NULL,
  eta_min = 1.001,
  progress = TRUE
)
}
\arguments{
\item{X}{An \eqn{n} x \eqn{d} matrix or data frame where \eqn{n} is the number of
observations and \eqn{d} is the number of variables.}

\item{G}{An integer vector specifying the numbers of clusters, which must be at least 1.}

\item{criterion}{A character string indicating the information criterion for model
selection. "BIC" is used by default. See the details section for a list of available
information criteria.}

\item{max_iter}{(optional) A numeric value giving the maximum number of
iterations each EM algorithm is allowed to use; 20 by default.}

\item{epsilon}{(optional) A number specifying the epsilon value for the
Aitken-based stopping criterion used in the EM algorithm: 0.01 by default.}

\item{init_method}{(optional) A string specifying the method to initialize
the EM algorithm. "kmedoids" clustering is used by default. Alternative
methods include "kmeans", "hierarchical", "mclust", and "manual". When "manual" is chosen,
a vector \code{clusters} of length \eqn{n} must be specified. If the data set is
incomplete, missing values will be first filled based on the mean imputation method.}

\item{clusters}{(optional) A numeric vector of length \eqn{n} that specifies the initial
cluster memberships of the user when \code{init_method} is set to "manual".
This argument is NULL by default, so that it is ignored whenever other given
initialization methods are chosen.}

\item{eta_min}{(optional) A numeric value close to 1 to the right specifying
the minimum value of eta; 1.001 by default.}

\item{progress}{(optional) A logical value indicating whether the
fitting progress should be displayed; TRUE by default.}
}
\value{
An object of class \code{MixtureMissing} with:
  \item{model}{The model used to fit the data set.}
  \item{pi}{Mixing proportions.}
  \item{mu}{Component location vectors.}
  \item{Sigma}{Component dispersion matrices.}
  \item{alpha}{Component proportions of good observations.}
  \item{eta}{Component degrees of contamination.}
  \item{z_tilde}{An \eqn{n} by \eqn{G} matrix where each row indicates the expected
    probabilities that the corresponding observation belongs to each cluster.}
  \item{v_tilde}{An \eqn{n} by \eqn{G} matrix where each row indicates the expected
    probabilities that the corresponding observation is good with respect
    to each cluster.}
  \item{clusters}{A numeric vector of length \eqn{n} indicating cluster
    memberships determined by the model.}
  \item{outliers}{A logical vector of length \eqn{n} indicating observations that are outliers.}
  \item{data}{The original data set if it is complete; otherwise, this is
    the data set with missing values imputed by appropriate expectations.}
  \item{complete}{An \eqn{n} by \eqn{d} logical matrix indicating which cells have no missing values.}
  \item{npar}{The breakdown of the number of parameters to estimate.}
  \item{max_iter}{Maximum number of iterations allowed in the EM algorithm.}
  \item{iter_stop}{The actual number of iterations needed when fitting the
    data set.}
  \item{final_loglik}{The final value of log-likelihood.}
  \item{loglik}{All the values of log-likelihood.}
  \item{AIC}{Akaike information criterion.}
  \item{BIC}{Bayesian information criterion.}
  \item{KIC}{Kullback information criterion.}
  \item{KICc}{Corrected Kullback information criterion.}
  \item{AIC3}{Modified AIC.}
  \item{CAIC}{Bozdogan's consistent AIC.}
  \item{AICc}{Small-sample version of AIC.}
  \item{ent}{Entropy.}
  \item{ICL}{Integrated Completed Likelihood criterion.}
  \item{AWE}{Approximate weight of evidence.}
  \item{CLC}{Classification likelihood criterion.}
  \item{init_method}{The initialization method used in model fitting.}
}
\description{
Carries out model-based clustering using a multivariate contaminated normal
mixture (MCNM). The function will determine itself if the data set is
complete or incomplete and fit the appropriate model accordingly. In the incomplete
case, the data set must be at least bivariate, and missing values are assumed to
be missing at random (MAR).
}
\details{
Available information criteria include
\itemize{
 \item AIC - Akaike information criterion
  \item BIC - Bayesian information criterion
  \item KIC - Kullback information criterion
  \item KICc - Corrected Kullback information criterion
  \item AIC3 - Modified AIC
  \item CAIC - Bozdogan's consistent AIC
  \item AICc - Small-sample version of AIC
  \item ICL - Integrated Completed Likelihood criterion
  \item AWE - Approximate weight of evidence
  \item CLC - Classification likelihood criterion
}
}
\examples{

data('auto')

#++++ With no missing values ++++#

X <- auto[, c('engine_size', 'city_mpg', 'highway_mpg')]
mod <- MCNM(X, G = 2, init_method = 'kmedoids', max_iter = 10)

summary(mod)
plot(mod)

#++++ With missing values ++++#

X <- auto[, c('normalized_losses', 'horsepower', 'highway_mpg', 'price')]
mod <- MCNM(X, G = 2, init_method = 'kmedoids', max_iter = 10)

summary(mod)
plot(mod)

}
\references{
Punzo, A. and McNicholas, P.D., 2016. Parsimonious mixtures of multivariate
  contaminated normal distributions. \emph{Biometrical Journal, 58}(6), pp.1506-1537. \cr \cr
Tong, H. and, Tortora, C., 2022. Model-based clustering and outlier detection
  with missing data. \emph{Advances in Data Analysis and Classification}.
}
