% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeContWeights.R
\name{makeContWeights}
\alias{makeContWeights}
\title{Continuous time weight estimation based on \code{aalen.predict}}
\usage{
makeContWeights(
  faFit,
  cfaFit,
  dataFr,
  atRiskState,
  eventState,
  startTimeName,
  stopTimeName,
  startStatusName,
  endStatusName,
  idName,
  b,
  weightRange = c(0, 10),
  willPlotWeights = TRUE,
  withLeftLim = FALSE
)
}
\arguments{
\item{faFit}{The \code{\link[timereg]{aalen}} fit for the factual hazard}

\item{cfaFit}{The  \code{\link[timereg]{aalen}} fit for the hypothetical hazard}

\item{dataFr}{\code{\link{data.frame}} or \code{\link{data.table}} on long format}

\item{atRiskState}{At risk state for the event of interest}

\item{eventState}{State for the event of interest}

\item{startTimeName}{Name of column with start time of the at risk period}

\item{stopTimeName}{Name of column with stop time of the at risk period}

\item{startStatusName}{Name of the column that contains the starting state for each interval}

\item{endStatusName}{Name of the column that contains the end state for each interval}

\item{idName}{Name of column in \code{dataFr} that identifies individuals}

\item{b}{Smoothing bandwidth parameter}

\item{weightRange}{Truncates weights outside this interval}

\item{willPlotWeights}{Plot indicator}

\item{withLeftLim}{Calculates left limit at jump if desired}
}
\value{
Longitudinally refined \code{\link{data.table}} of the initial \code{dataFr} with \code{weights} column added.
}
\description{
Refines data longitudinally in order to do estimate parameters(e.g. Nelson-Aalen or Kaplan-Meier) using continuous time weights. The weights can be assessed using the plot option.
}
\examples{
library(data.table)
library(timereg)

# fr1 is a longitudinal data set with subjects that are diagnosed at time 0, and may
# be treated as time evolves. Subjects can die before receiving treatment

# The method assumes there are no tied event times in the observed data. Although there are no
# tied event times in fr1, we use the function addNoiseAtEventTimes() for illustration here
fr1 <- as.data.table(fr1)
fr1 <- addNoiseAtEventTimes(fr1)

# Time to treatment and death are confounded by the baseline variable L. We want to
# mimic a scenario where time to treatment is randomized (and does not depend on L):
fr1_diag <- fr1[fr1$from.state == "diag", ]
fFit <- aalen(
  Surv(from, to, to.state == "treat") ~ 1 + L, data = fr1_diag, n.sim = 50L,
  robust = 0
)
cfFit <- aalen(
  Surv(from, to, to.state == "treat") ~ 1, data = fr1_diag, n.sim = 50L,
  robust = 0
)

# We calculate and plot the weights
frame1 <- makeContWeights(fFit, cfFit, fr1, "diag", "treat", "from", "to",
  "from.state", "to.state", "id",
  b = 0.4,
  weightRange = c(0, 5)
)

# We fit a weighted model for the outcome. A is a treatment indicator (A=1 means treated).
a1 <- aalen(
  Surv(from, to, to.state == "death") ~ 1 + A, data = frame1,
  weights = frame1$weights, n.sim = 50L, robust = 0
)

# We plot the A coefficient from the weighted regression,
# and compare with the true hypothetical coefficient
plot(a1$cum[, c(1, 3)],
  type = "s", ylim = c(-1.2, 0.5), xlim = c(0, 5),
  main = "Weighted additive hazard treatment coefficient"
)
lines(Tmat, col = 2)
legend("bottomleft", c("weighted estimate", "hypothetical treatment coef"),
  lty = 1, col = c(1, 2), bty = "n"
)

# Next we consider an example with dependent censoring.
# Subjects are censored depending on a baseline variable u. We wish to mimic the
# cumulative hazard for death we would have seen if the censoring were independent.

faFit <- aalen(
  Surv(from, to, to.state == "Censored") ~ 1 + u, data = fFrame, n.sim = 50L,
  robust = 0
)
cfaFit <- aalen(
  Surv(from, to, to.state == "Censored") ~ 1, data = fFrame, n.sim = 50L,
  robust = 0
)

frame <- makeContWeights(
  faFit, cfaFit, fFrame, "Alive", "Censored", "from", "to", "from.state",
  "to.state", "id", 100
)

fMod <- aalen(
  Surv(from, to, to.state == "Dead") ~ 1, data = fFrame, n.sim = 50L,
  robust = 0
)
wMod <- aalen(
  Surv(from, to, to.state == "Dead") ~ 1, data = frame, weights = frame$weights,
  n.sim = 50L, robust = 0
)

plot(fMod$cum, type = "s", main = "Nelson-Aalen for death", ylab = "")
lines(wMod$cum, type = "s", col = "red")
legend("topleft", c("factual", "weighted factual"), lty = 1, col = c(1, "red"), bty = "n")
}
\references{
\url{https://arxiv.org/abs/1802.01946}
}
\author{
Pål Christie Ryalen <p.c.ryalen@medisin.uio.no>
}
