% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assign_traffic.R
\name{assign_traffic}
\alias{assign_traffic}
\title{Algorithms for solving the Traffic Assignment Problem (TAP).}
\usage{
assign_traffic(
  Graph,
  from,
  to,
  demand,
  algorithm = "bfw",
  max_gap = 0.001,
  max_it = .Machine$integer.max,
  aon_method = "bi",
  constant = 1,
  dial_params = NULL,
  verbose = TRUE
)
}
\arguments{
\item{Graph}{An object generated by \link{makegraph} function.}

\item{from}{A vector of origins}

\item{to}{A vector of destinations.}

\item{demand}{A vector describing the flow between each origin-destination pair.}

\item{algorithm}{character. \code{msa}, \code{fw}, \code{cfw}, \code{bfw} or \code{dial}. Default to \code{bfw}. See details.}

\item{max_gap}{Numeric. Relative gap to achieve. Default to 0.001.}

\item{max_it}{Numeric. Maximum number of iterations. Default to \code{.Machine$integer.max}}

\item{aon_method}{Character.\code{d}, \code{bi}, \code{nba}, \code{cphast} or \code{cbi}. Default to \code{bi}. See details.}

\item{constant}{numeric. Constant to maintain the heuristic function admissible in NBA* algorithm. Default to 1, when cost is expressed in the same unit than coordinates. See details}

\item{dial_params}{List. Named list of hyperparameters for \code{dial} algorithm. See details.}

\item{verbose}{Logical. If \code{TRUE} (default), progression is displayed.}
}
\value{
A \code{list} containing : \itemize{
 \item The relative gap achieved
 \item Number of iteration
 \item A data.frame containing edges attributes, including equilibrated flows, new costs and free-flow travel times.

 }
}
\description{
Estimation of the User Equilibrium (UE)
}
\details{
The most well-known assumptions in traffic assignment models are the ones following Wardrop's first principle.
Traffic assignment models are used to estimate the traffic flows on a network. These models take as input a matrix of flows that indicate the volume of traffic between origin and destination (O-D) pairs.
Unlike All-or-Nothing assignment (see \link{get_aon}), edge congestion is modeled through the \strong{Volume Decay Function (VDF)}.
The Volume Decay Function used is the most popular in literature, from the Bureau of Public Roads :

\strong{t = t0 * (1 + a * (V/C)^b) }
with t = actual travel time (minutes),
t0 = free-flow travel time (minutes),
a = alpha parameter (unitless),
b = beta parameter (unitless),
V = volume or flow (veh/hour)
C = edge capacity (veh/hour)

Traffic Assignment Problem is a convex problem and solving algorithms can be divided into two categories : \itemize{
\item link-based : \strong{Method of Successive Average} (\code{msa}) and \strong{Frank-Wolfe variants} (normal : \code{fw}, conjugate : \code{cfw} and bi-conjugate : \code{bfw}).
These algorithms uses the descent direction given by AON assignment at each iteration, all links are updated at the same time.
\item bush-based : \strong{Algorithm-B} (\code{dial})
The problem is decomposed into sub-problems, corresponding to each origin of the OD matrix, that operate on acyclic sub-networks of the original transportation network, called bushes.
Link flows are shifted from the longest path to the shortest path recursively within each bush using Newton method.
}

Link-based algorithms are historically the first algorithms developed for solving the traffic assignment problem. It require low memory and are known to tail in the vicinity of the optimum and usually cannot be used to achieve highly precise solutions.
Algorithm B is more recent, and is better suited for achieve the highest precise solution. However, it require more memory and can be time-consuming according the network size and OD matrix size.
In \code{cppRouting}, the implementation of algorithm-B allow "batching", i.e. bushes are temporarily stored on disk if memory limit, defined by the user, is exceeded.
Please see the package website for practical example and deeper explanations about algorithms. (\url{https://github.com/vlarmet/cppRouting/blob/master/README.md})

Convergence criterion can be set by the user using max_gap argument, it is the relative gap which can be written as :
\strong{abs(TSTT/SPTT - 1)}
with TSTT (Total System Travel Time) = sum(flow * cost),
SPTT (Shortest Path Travel Time) = sum(aon * cost)

Especially for link-based algorithms (msa, *fw), the larger part of computation time rely on AON assignment. So, choosing the right AON algorithm is crucial for fast execution time.
Contracting the network on-the-fly before AON computing can be faster for large network and/or large OD matrix.

AON algorithms are : \itemize{
\item \code{bi} : bidirectional Dijkstra algorithm
\item \code{nba} : bidirectional A* algorithm, nodes coordinates and constant parameter are needed
\item \code{d} : Dijkstra algorithm
\item \code{cbi} : contraction hierarchies + bidirectional search
\item \code{cphast} : contraction hierarchies + phast algorithm
}
These AON algorithm can be decomposed into two families, depending the sparsity of origin-destination matrix : \itemize{
\item recursive pairwise : \code{bi}, \code{nba} and \code{cbi}. Optimal for high sparsity. One-to-one algorithm is called N times, with N being the length of from.
\item recursive one-to-many : \code{d} and \code{cphast}. Optimal for dense matrix. One-to-many algorithm is called N times, with N being the number of unique from (or to) nodes
}

For large instance, it may be appropriate to test different \code{aon_method} for few iterations and choose the fastest one for the final estimation.

Hyperparameters for algorithm-b are : \itemize{
\item \code{inneriter} : number of time bushes are equilibrated within each iteration. Default to 20
\item \code{max_tol} : numerical tolerance. Flow is set to 0 if less than max_tol. Since flow shifting consist of iteratively adding or substracting double types, numerical error can occur and stop convergence.
Default to 1e-11.
\item \code{tmp_path} : Path for storing bushes during algorithm-B execution. Default using \code{tempdir()}
\item \code{max_mem} : Maximum amount of RAM used by algorithm-B in gigabytes. Default to 8.

}


In New Bidirectional A star algorithm, euclidean distance is used as heuristic function.
To understand the importance of constant parameter, see the package description : \url{https://github.com/vlarmet/cppRouting/blob/master/README.md}
All algorithms are partly multithreaded (AON assignment).
}
\note{
\code{from}, \code{to} and \code{demand} must be the same length.
\code{alpha}, \code{beta} and \code{capacity} must be filled in during network construction. See \link{makegraph}.
}
\examples{
#Choose number of cores used by cppRouting
RcppParallel::setThreadOptions(numThreads = 1)

#Data describing edges of the graph
edges<-data.frame(from_vertex=c(0,0,1,1,2,2,3,4,4),
                  to_vertex=c(1,3,2,4,4,5,1,3,5),
                  cost=c(9,2,11,3,5,12,4,1,6))

# Origin-destination trips
trips <- data.frame(from = c(0,0,0,0,1,1,1,1,2,2,2,3,3,4,5,5,5,5,5),
                    to = c(1,2,5,3,2,5,2,4,2,5,2,3,5,2,0,0,3,5,1),
                    flow = c(10,30,15,5,5,2,3,6,4,15,20,2,3,6,2,1,4,5,3))

#Construct graph
graph <- makegraph(edges,directed=TRUE, alpha = 0.15, beta = 4, capacity = 5)


# Solve traffic assignment problem
## using Bi-conjugate Frank-Wolfe algorithm
traffic <- assign_traffic(Graph=graph,
                          from=trips$from, to=trips$to, demand = trips$flow,
                          algorithm = "bfw")
print(traffic$data)

## using algorithm-B
traffic2 <- assign_traffic(Graph=graph,
                           from=trips$from, to=trips$to, demand = trips$flow,
                           algorithm = "dial")
print(traffic2$data)
}
\references{
Wardrop, J. G. (1952). "Some Theoretical Aspects of Road Traffic Research".

M. Fukushima (1984). "A modified Frank-Wolfe algorithm for solving the traffic assignment problem".

R. B. Dial (2006). "A path-based user-equilibrium traffic assignment algorithm that obviates path storage and enumeration".

M. Mitradjieva, P. O. Lindberg (2012).  "The Stiff Is Moving — Conjugate Direction Frank-Wolfe Methods with Applications to Traffic Assignment".
}
