#' ctStanPlotPost
#'
#' Plots prior and posterior distributions of model parameters in a ctStanModel or ctStanFit object.
#' 
#' @param obj fit or model object as generated by \code{\link{ctStanFit}},
#' \code{\link{ctModel}}, or \code{\link{ctStanModel}}.
#' @param rows vector of integers denoting which rows of ctstanmodel$pars to plot priors for. 
#' Character string 'all' plots all rows with parameters to be estimated. 
#' @param mfrow 2 dimensional integer vector defining number of rows and columns of plots,
#' as per the mfrow argument to \code{\link[graphics]{par}}.
#' 'auto' determines automatically, to a maximum of 4 by 4, while \code{NULL} 
#' uses the current system setting.
#' @param parcontrol parameters to pass to \code{\link[graphics]{par}} which temporarily
#' change plot settings.
#' @param wait If true, user is prompted to continue before plotting next graph.  
#' If false, graphs are plotted one after another without waiting.
#' @examples
#' ctStanPlotPost(ctstantestfit)
#' @export

ctStanPlotPost<-function(obj, rows='all',mfrow='auto',
  parcontrol=list(mgp=c(1.3,.5,0),mar=c(3,2,2,1)+.2),wait=FALSE){

  if(!(class(obj) %in% c('ctStanFit','ctStanModel'))) stop('not a ctStanFit or ctStanModel object!')
  if(class(obj)=='ctStanFit') dopost <- TRUE else dopost <- FALSE
  
  if(dopost) sm<-obj$ctstanmodel else sm <- obj
  
  #get unique parameters
  m <- sm$par[match(unique(sm$par$param), sm$par$param),]
  m <- m[!is.na(m$param),] #remove non free param
  m <- m[!(m$param %in% 'stationary'),] #remove stationary params
  
  paroriginal<-graphics::par()[c('mfrow','mgp','mar')]
  
  do.call(graphics::par,parcontrol)
  
  if(dopost) s<-rstan::extract(obj$stanfit)
  
  if(rows=='all') rows<-(1:nrow(m))[is.na(m$value)]  
  if(!is.null(mfrow)){
    if(mfrow=='auto') {
      graphics::par(mfrow=c(min(3,grDevices::n2mfrow( (length(rows)+sum(m$indvarying)*3))[2]), 
        min(3,n2mfrow( (length(rows)+sum(m$indvarying)*3))[1])))
    }
    if(mfrow!='auto') graphics::par(mfrow=mfrow)
  }
  
  
  if(dopost){
  snames<-names(s)
  hmeannames<-snames[grep('hmean_',snames)]
  hsdnames<-snames[grep('hsd_',snames)]
  tipnames<-snames[grep('tipred_',snames)]
  sum<-getMethod('summary','stanfit')(obj$stanfit)
  sumnames<-dimnames(sum$summary)[[1]]
  nsubjects<-max(  as.numeric(unlist(lapply(grep('^indparams[[]',sumnames),function(x){
    strsplit(gsub("[^0-9,]", "", sumnames[x]), ",")[[1]][1]
  }))))
  if(nsubjects < 1) nsubjects<-1
  }
  
  
  indvaryingcount<-0
  hypermeancount<-0
  # pmeans<-matrix(NA,nrow=nsubjects,ncol=sum(m$indvarying))
  # pnames<-rep(NA,sum(m$indvarying))
  for(rowi in rows){
    hypermeancount <- which(m$param[is.na(m$value)] %in% m$param[rowi])
    
    p<-list(lwd=2, #plot pars
      x=NA,
      xaxs='i',
      yaxs='i',
      xlab='Value', 
      main=paste0('Pop. mean ',m$param[rowi])) 
    
    if(dopost){
    hypermean<- s$hypermeans[,hypermeancount]
    
    param<-hypermean
    dhypermeanpost<-ctDensity(eval(parse(text=paste0(m$transform[rowi]))))
    
    p$xlim=dhypermeanpost$xlim
    p$ylim=c(0,dhypermeanpost$ylim[2])
    p$x = dhypermeanpost$density
    }
    
    param<-stats::rnorm(50000,0,1)
    meanprior <- eval(parse(text=paste0(m$transform[rowi])))
    dmeanprior<-ctDensity(meanprior)
    
    if(!dopost){
      p$x = NA
      p$xlim = dmeanprior$xlim
      p$ylim=dmeanprior$ylim
    }
    
    do.call(graphics::plot,p) 
    
    # meanprior <- meanprior[meanprior > dsubjectprior$xlim[1]-5 & meanprior < dsubjectprior$xlim[2] + 5]
    if(dopost) leg <- c('Pop. mean posterior','Pop. mean prior') else leg <- 'Pop. mean prior'
    if(dopost) legcol <- c('black','blue') else legcol <- 'black'
    
    graphics::points(dmeanprior$density,col=ifelse(dopost,'blue','black'),type='l',lty=ifelse(dopost,2,1),lwd=2)
    graphics::legend('topright',leg,
      text.col=legcol,bty='n')
    
    
    if(m$indvarying[rowi]){ #then also plot sd and subject level pars
      
      indvaryingcount<-which(m$param[is.na(m$value) & m$indvarying] %in% m$param[rowi])
      
      sdscale <- m$sdscale[rowi]
      tform <- gsub('.*', '*',sm$hypersdtransform,fixed=TRUE)
      
      if(dopost) {
        rawhypersd<-s$rawhypersd[,indvaryingcount] 
      
      hypersd <- eval(parse(text=tform))  #hypersd samples

      
      indparams<-s[['indparams']][,1:nsubjects,indvaryingcount]
      
      param<-indparams
      dindparams<-ctDensity(eval(parse(text=paste0(m$transform[rowi]))))
      param<-stats::rnorm(50000,hypermean,hypersd)
      dsubjectprior<-ctDensity(eval(parse(text=paste0(m$transform[rowi]))))
      param<-stats::rnorm(50000,0,1)
      
      graphics::plot(dindparams$density,xlab='Value',main=m$param[rowi],lwd=2,yaxs='i',
        ylim=c(0,max(dsubjectprior$ylim[2],dindparams$ylim[2])),
        xlim=c(min(dsubjectprior$xlim[1],dindparams$xlim[1]),max(dsubjectprior$xlim[2],dindparams$xlim[2])))
      graphics::points(dsubjectprior$density,col='red',lwd=2,lty=2,type='l')
      graphics::points(dmeanprior$density,lwd=2,type='l',col='blue',lty=3)
      graphics::legend('topright',c('Subject param posterior','Subject param prior','Pop mean prior'),
        text.col=c('black','red','blue'),bty='n')
      
      }
      
      
      
      #pre-transform hyper std dev
      p<-list(lwd=2, #plot pars
        x=NA,
        xaxs='i',
        yaxs='i',
        xlab='Value', 
        main=paste0('Pre-tform pop. sd ',m$param[rowi])) 
      
      #posterior
      if(dopost) {
        dhypersdpost<-ctDensity(hypersd)
        p$xlim=c(0,dhypersdpost$xlim[2])
        p$ylim=c(0,dhypersdpost$ylim[2])
        p$x=dhypersdpost$density
      }
      #prior
      rawhypersd<-  stats::rnorm(100000,0,1)
      hypersdprior <- eval(parse(text=tform)) #hypersd prior samples
      dhypersdprior<-ctDensity(hypersdprior)
      
      if(!dopost) {
        p$xlim=c(0,dhypersdprior$xlim[2])
        p$ylim=c(0,dhypersdprior$ylim[2])
      }
      
      do.call(graphics::plot,p)
      graphics::points(dhypersdprior$density,col=ifelse(dopost,'blue','black'),type='l',lty=ifelse(dopost,2,1),lwd=2)
      
      if(dopost) leg <- c('Pop. sd posterior','Pop. sd prior') else leg <- 'Pop. sd prior'
      if(dopost) legcol <- c('black','blue') else legcol <- 'black'
      
      graphics::legend('topright',leg,text.col=legcol,bty='n')
      
      
      
      
      #post-transform hyper std dev
      p<-list(lwd=2, #plot pars
        x=NA,
        xaxs='i',
        yaxs='i',
        xlab='Value', 
        main=paste0('Pop. sd ',m$param[rowi])) 
      
      #posterior
      if(dopost){
      hsdpost <- s[[paste0('hsd_',m$param[rowi])]]
      dhsdpost<-ctDensity(hsdpost)

      p$x = dhsdpost$density
      p$ylim=c(0,dhsdpost$ylim[2])
      p$xlim=c(0, dhsdpost$xlim[2])
      }
      
      if(!dopost) hypermean <- rnorm(length(hypersdprior),0,1) #otherwise the prior is plotted conditional on sampled hypermeans
      param<-suppressWarnings(hypermean+hypersdprior)
      high<-eval(parse(text=paste0(m$transform[rowi])))
      param<-suppressWarnings(hypermean-hypersdprior)
      low<-eval(parse(text=paste0(m$transform[rowi])))
      hsdprior<-abs(high - low)/2
      
      dhsdprior<-ctDensity(sample(hsdprior,50000,replace=TRUE))
      
      if(!dopost) {
        p$xlim=c(0,dhsdprior$xlim[2])
        p$ylim=c(0,dhsdprior$ylim[2])
      }
      
      
      do.call(graphics::plot,p)
      graphics::points(dhsdprior$density,col=ifelse(dopost,'blue','black'),type='l',lty=ifelse(dopost,2,1),lwd=2)
      
      if(dopost) leg <- c('Pop. sd posterior','Pop. sd prior') else leg <- 'Pop. sd prior'
      if(dopost) legcol <- c('black','blue') else legcol <- 'black'
      
      graphics::legend('topright',leg, text.col=legcol, bty='n')
      

      
      
      
      # pmeans[,indvaryingcount]<-sum$summary[sumnames %in% 
      #     paste0(m$matrix[rowi],'[',1:nsubjects,',',m$row[rowi],
      #       if(!(m$matrix[rowi] %in% c('T0MEANS','CINT','MANIFESTMEANS'))) paste0(',',m$col[rowi]),']'),
      #   1]
      # 
      # pnames[indvaryingcount]<-paste0(m$matrix[rowi],'[',m$row[rowi],',',m$col[rowi],']')
      
      
      if(wait==TRUE & rowi != utils::tail(rows,1)){
        message("Press [enter] to display next plot")
        readline()
      }
    }
  }
  
  # if(length(rows)>1){
  #   require(lattice)
  #   colnames(pmeans)<-pnames
  #   splom(pmeans,
  #     upper.panel=function(x,y) panel.loess(x, y),
  #     diag.panel = function(x,...){
  #       yrng <- current.panel.limits()$ylim
  #       d <- density(x, na.rm=TRUE)
  #       d$y <- with(d, yrng[1] + 0.95 * diff(yrng) * y / max(y) )
  #       panel.lines(d)
  #       diag.panel.splom(x,...)
  #     },
  #     lower.panel = function(x, y, i,j){
  #       panel.splom(x,y,col=rgb(0,0,1,alpha=.2),pch=16)
  #       
  #       if( j==1)  panel.axis(side=c('left'),outside=T,
  #         at=round(seq(max(y),min(y),length.out=5)[c(-1,-5)],digits=3),tck=1,
  #         labels=round(seq(max(y),min(y),length.out=5)[c(-1,-5)],digits=3))
  #       
  #       if( i==ncol(pmeans))  panel.axis(side=c('bottom'),outside=T,
  #         at=round(seq(max(x),min(x),length.out=5)[c(-1,-5)],digits=3),tck=1,
  #         labels=round(seq(max(y),min(y),length.out=5)[c(-1,-5)],digits=3))
  #     },
  #     pscales=0,
  #     as.matrix = TRUE,
  #     varname.cex=.4,
  #     strip=T,
  #     main='Subject level parameter plots',
  #     xlab=NULL,ylab=NULL,
  #     col=rgb(0,0,1,alpha=.2),pch=16
  #   )
  # }
  
  do.call(graphics::par,paroriginal)
  
}

