% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hasserls.R
\encoding{UTF-8}
\name{hasserls}
\alias{hasserls}
\title{Hasse diagram of the restricted layout structure}
\usage{
hasserls(
  object,
  randomisation.objects,
  random.arrows = NULL,
  showRLS = "Y",
  showpartialRLS = "Y",
  showdfRLS = "Y",
  showrandRLS = "Y",
  check.confound.df = "Y",
  maxlevels.df = "Y",
  table.out = "N",
  equation.out = "N",
  pdf = "N",
  example = "example",
  outdir = NULL,
  hasse.font = "sans",
  produceBWPlot = "N",
  structural.colour = "grey",
  structural.width = 2,
  partial.colour = "orange",
  partial.width = 1.5,
  objects.colour = "mediumblue",
  df.colour = "red",
  arrow.colour = "mediumblue",
  arrow.width = 1.5,
  arrow.pos = 7.5,
  larger.fontlabelmultiplier = 1,
  middle.fontlabelmultiplier = 1,
  smaller.fontlabelmultiplier = 1
)
}
\arguments{
\item{object}{An object of class \code{"rls"}. The function \code{\link[hassediagrams]{itemlist}} generates the class \code{"rls"} object. 
Printing the "\code{"rls"} object will give a list of structural objects (that define the layout structure) to aid in 
defining the randomisation objects in the restricted layout structure.}

\item{randomisation.objects}{This argument takes the format of the \code{TransferObject} item from the class \code{"rls"} object. 
The first column contains the names of all the structural objects in the layout structure (automatically generated) and the 
second column contains the corresponding randomisation objects in the restricted layout structure (manually generated). 
To begin with, the function \code{\link[hassediagrams]{itemlist}} should be run to generate the class \code{"rls"} object. 
The user will then need to edit the second column of the \code{TransferObject} matrix to define the randomisation objects 
that appear in the restricted layout structure. Structural objects that do not occur in the restricted layout structure must be 
left as "NULL" in the second column. The names specified in the second column represent the labels of the randomisation objects 
on the Hasse diagram of the restricted layout structure.}

\item{random.arrows}{A matrix of two columns that takes integer entries. The entries in the first column contain the object(s) 
at the start of the randomisation arrow and the second column contains the object(s) at the end. 
The values correspond to the entry number for the randomisation object in the \code{TransferObject} 
item from the class \code{"rls"} object). The first column specifies the randomisation object(s) at the beginning of the 
randomisation arrow(s) and the second column specifies the randomisation object(s) at the end of the arrow. 
The randomisation arrows must point downwards, hence, in each row of the matrix the second column entry must be larger than the first column entry. 
The randomisation object(s) involved in the randomisation arrow(s) must first be specified in the randomisation.objects argument.}

\item{showRLS}{logical. If "N" then generation of the Hasse diagram of the restricted layout structure is suppressed. The default is "Y".}

\item{showpartialRLS}{logical. If "N" then the partial crossing between randomisation objects (using dotted connecting lines) is not illustrated
on the Hasse diagram of the restricted layout structure. The default is "Y".}

\item{showdfRLS}{logical. If "N" then the randomisation object label is not displayed on the Hasse diagram of the restricted layout structure. The default is "Y".}

\item{showrandRLS}{logical. If "N" then the randomisations are not illustrated (using arrows) on the Hasse diagram of the restricted layout structure. 
The default is "Y". If \code{random.arrows=NULL}, then \code{showrandRLS} defaults to "N".}

\item{check.confound.df}{logical. If "N" then the check for confounded degrees of freedom is not performed. The default is "Y".}

\item{maxlevels.df}{logical. If "N" then the potential maximum number of levels of a generalised factor is removed from the randomisation object 
label on the Hasse diagram of the restricted layout structure. The default is "Y".}

\item{table.out}{logical. If "Y" then a table that shows the relationships between the randomisation objects in the restricted layout structure is printed. 
The default is "N".}

\item{equation.out}{logical. If "Y" then a recommended mixed model to use in the statistical analysis is printed. The default is "N".}

\item{pdf}{logical. If "Y" then a pdf file containing the Hasse diagram of the restricted layout structure is generated. 
The default is "N", i.e., a pdf file is not generated.}

\item{example}{character. Filename for the pdf output file containing the Hasse diagram. The default is set to "example".}

\item{outdir}{character. Location of the pdf output file if \code{pdf="Y"}. The default is set to \code{NULL} and in this case the pdf output file 
containing the Hasse diagram output is stored to a temporary file. To specify a permanent location this argument needs be specified.}

\item{hasse.font}{character. The name of the font family used for all text included on the Hasse diagram. 
Standard and safe font families to choose are "sans", "serif", and "mono". 
If any of the labels of the randomisation objects (as defined in the second column of \code{randomisation.objects} matrix) 
contain Unicode characters, a Unicode friendly font family should be selected. 
For more details on Unicode friendly family options see the Details section. 
If the font family selected fails to render, the font is automatically changed to "sans" instead.
The default is "sans".}

\item{produceBWPlot}{logical. If "Y" then the Hasse diagram will be generated in black and white format. 
The default is set to "N", i.e., a coloured version of the plot is produced.}

\item{structural.colour}{character. The colour of the structural lines that connect randomisation objects on the Hasse diagram. The default colour is "grey".}

\item{structural.width}{numeric. The width of the structural lines on the Hasse diagram. The default width is 2.}

\item{partial.colour}{character. The colour of the partial crossing dotted lines of the connecting randomisation objects on the Hasse diagram. 
The default colour is "orange".}

\item{partial.width}{numeric. The width of the partial crossing dotted lines on the Hasse diagram. The default width is 1.5.}

\item{objects.colour}{character. The colour of the labels of the randomisation objects on the Hasse diagram. The default colour is "mediumblue".}

\item{df.colour}{character. The colour of the degrees of the freedom labels on the Hasse diagram. The default colour is "red".}

\item{arrow.colour}{character. The colour of the randomisation arrows on the Hasse diagram. The default colour is "mediumblue".}

\item{arrow.width}{numeric. The randomisation arrows width on the Hasse diagram. The default width is 1.5.}

\item{arrow.pos}{numeric. Specifies the position of the randomisation arrows, i.e., how far the randomisation arrows will be from 
the objects they point at. The default is 7.5. A smaller number specifies longer arrows and a higher number specifies shorter arrows.}

\item{larger.fontlabelmultiplier}{numeric. The large font multiplier is the multiplier for the font used for the labels of objects on the 
Hasse diagram where there are four or less objects at that level in the diagram. The default is 1.}

\item{middle.fontlabelmultiplier}{numeric. The medium font multiplier is the multiplier for the font used for the labels of objects on the 
Hasse diagram involving a factor that is equivalent to a generalised factor. The default is 1.}

\item{smaller.fontlabelmultiplier}{numeric. The small font multiplier is the multiplier for the font used for the labels of objects on the 
Hasse diagram where there are five or more objects at that level of the diagram. The default is 1.}
}
\value{
The function \code{\link[hassediagrams]{hasserls}} returns:
1. The Hasse diagram of the restricted layout structure (if \code{showRLS = "Y"}).

2. The restricted layout structure table shows the relationships between the randomisation objects in the restricted layout structure 
(if \code{table.out="Y"}). The individual entries in the table consist of blanks on the main diagonal and 0’s, (0)’s or 1’s elsewhere. 
If the factor (or generalised factor) corresponding to the ith row and the factor (or generalised factor) corresponding to the jth column are fully crossed, 
then a 0 is entered in the (i,j)th entry in the table. If these factors (or generalised factors) are partially crossed, or the ith row factor 
(or generalised factor) only has one level and nests the jth column factor (or generalised factor), then the (i,j)th entry is (0). 
If the ith row factor (or generalised factor) is nested within the jth column factor (or generalised factor), then a 1 is entered in the 
(i,j)th entry. If two factors (or generalised factor) are equivalent, then they share a single row and column in the table, 
where the row and column headers include both factor (or generalised factor) names, separated by an "=" sign.

3. An equation that suggests the mixed model to be fitted (if \code{equation.out="Y"}).

4. If there are confounded degrees of freedom, a table of the structural objects and a description of the associated degrees of freedom is printed.
}
\description{
Returns a Hasse diagram of the restricted layout structure of an experimental design
}
\details{
The \code{\link[hassediagrams]{hasserls}} function generates the Hasse diagram of the restricted layout structure. 
The Hasse diagram consists of a set of randomisation objects, corresponding to the factors and generalised factors, 
and the relationships between the objects (either crossed, nested, partially crossed or equivalent), 
as defined by the structure of the experimental design and the randomisation performed, see Bate and Chatfield (2016b).

The function requires an object of class \code{"rls"} containing the structural objects in the layout structure.

Where present, two partially crossed factors are illustrated on the diagram with a dotted line connecting them. 
This feature can be excluded using the \code{showpartialRLS} option.

The maximum number of possible levels of each generalised factor, along with the actual number present in the design 
and the "skeleton ANOVA" degrees of freedom, can be included in the randomisation object label on the Hasse diagram.

The randomisation arrows that illustrate the randomisation performed can be included on the Hasse diagram.

The \code{\link[hassediagrams]{hasserls}} function evaluates the design in order to identify if there are any 
confounded degrees of freedom across the design. It is not recommended to perform this evaluation for large designs, 
due to the potential high computational cost. This can be controlled using the \code{check.confound.df = "N"} option. 


The rendering of Unicode symbols (e.g., u2297, u2192) in the Hasse diagram depends on the operating system and the font selected in hasse.font.

macOS / Linux:
Most system fonts that support Unicode work directly in plotting and PDF output without explicit registration. In many cases, the default "sans" family is sufficient for PDF rendering of these symbols.
However, for on-screen rendering usually unicode-friendly fonts like "AppleMyungjo", "Arial Unicode MS", .SF Compact, and "Noto Sans Math" are needed. 

Windows:
Base R plotting often requires explicit font registration before the font can be used. Even if the font is installed, it may not be accessible to the graphics device until registered.
Unicode-friendly fonts are "Lucida Sans Unicode", "Arial Unicode MS", "Segoe UI Symbol", "Cambria", "Noto Sans Math" and "Ebrima". 
The aforementioned fonts may not not be available in your R session. The available system fonts can be printed by systemfonts::system_fonts()$family.
System available fonts can be imported by running showtext::font_import() or extrafont::font_import().
To check which fonts have been successfully imported, run showtext::fonts() or extrafont::fonts().
The Arial Unicode MS font can be downloaded from online sources.
The Noto Sans Math font can be installed using sysfonts::font_add_google("Noto Sans Math").
For Windows, fonts might not be accessible to the graphics device until registered using:
windowsFonts(LucidaSansUnicode = windowsFont("Lucida Sans Unicode"))
adapted to the font need to use.
}
\examples{
\donttest{
## NOTE TO USERS:
## Some examples below need Unicode symbols (e.g., "u2297 and "u2192"
## with a backslash, for the Kronecker and arrow symbols respectively),
## but we use ASCII fallbacks such as "x" and "->" in the code to ensure
## compatibility across systems.
## To render proper Unicode symbols in diagrams, update the labels manually
## i.e., x to \ u2297 to get the Kronecker symbol and -> to \ u2192 to get the
## arrow symbol, and set a Unicode-friendly font via the hasse.font argument.
## See the hasse.font argument and Details section in ?hasserls for guidance.

## Example: Asphalt concrete production (fractional factorial design)
concrete_objects <- itemlist(datadesign = concrete)
concrete_rls <- concrete_objects$TransferObject
concrete_rls[, 2] <- concrete_rls[, 1]
concrete_rls[27, 2] <- "AC^AG^CC^CoT^CuT -> Run"  
hasserls(object = concrete_objects,
         randomisation.objects = concrete_rls,
         larger.fontlabelmultiplier = 1.6,
         smaller.fontlabelmultiplier = 1.3,
         table.out = "Y", arrow.pos = 8,
         hasse.font = "sans")

## Example: Crossover dental study
dental_objects <- itemlist(datadesign = dental, randomfacsid = c(0,1,0,0,0))
dental_rand_arrows <- matrix(c(3, 5, 4, 7), ncol = 2, byrow = TRUE)
dental_rls <- dental_objects$TransferObject
dental_rls[c(1:5,7,8), 2] <- c("Mean", "Period", "Sequence",
                               "Treatment", "Subject[Sequence]",
                               "Period x Sequence",
                               "Observation")
hasserls(object = dental_objects,
         randomisation.objects = dental_rls,
         random.arrows = dental_rand_arrows,
         larger.fontlabelmultiplier = 1.6,
         table.out = "Y", equation.out = "Y",
         arrow.pos = 15, hasse.font = "sans")

## Example: Human-computer interaction study
human_objects <- itemlist(datadesign = human,
                          randomfacsid = c(1,1,0,0,0,0,1))
human_rand_arrows <- matrix(c(3, 12, 6, 13), ncol = 2, byrow = TRUE)
human_rls <- human_objects$TransferObject
human_rls[, 2] <- c("Mean", "Day", "Method", "Period", "Room", "Sequence",
                    "NULL", "Subject | Subject -> Day x Sequence",  
                    "NULL", "NULL", "NULL", "Day x Room",
                    "Period x Room", "NULL",
                    "Test = Day^Method^Period^Room^Sequence")
hasserls(object = human_objects,
         randomisation.objects = human_rls,
         random.arrows = human_rand_arrows,
         larger.fontlabelmultiplier = 1.4,
         hasse.font = "sans")

## Example: Analytical method (cross-nested design)
analytical_objects <- itemlist(datadesign = analytical,
                               randomfacsid = c(0,0,1,1,1,0,0,0))
analytical_rand_arrows <- matrix(c(2, 19, 19, 20), ncol = 2, byrow = TRUE)
analytical_rls <- analytical_objects$TransferObject
analytical_rls[, 2] <- c("Mean", "Batch", "Site",
                         "Analyst[Site]", "Column[Site]",
                         "System[Site]", "NULL",
                         "{Analyst^Column}[Site]",
                         "{Analyst^System}[Site]", "NULL",
                         "{Column^System}[Site]",
                         "NULL", "NULL",
                         "{Analyst^Column^System}[Site] -> Run",  
                         "NULL", "NULL", "NULL", "NULL",
                         "Preparation[Run]", "Injection[Run]")
hasserls(object = analytical_objects,
         randomisation.objects = analytical_rls,
         random.arrows = analytical_rand_arrows,
         showpartialRLS = "N", check.confound.df = "N",
         larger.fontlabelmultiplier = 1,
         smaller.fontlabelmultiplier = 1.6,
         hasse.font = "sans")

## Conditionally run examples requiring 'dae'
if (requireNamespace("dae", quietly = TRUE)) {
  data(BIBDWheat.dat, package = "dae")
  BIBDWheat <- BIBDWheat.dat[, -4]
  BIBDWheat$Plots <- 1:30
  BIBDWheat_objects <- itemlist(datadesign = BIBDWheat)
  IBDWheat_rand_arrows <- matrix(c(3, 4), ncol = 2, byrow = TRUE)
  IBDWheat_rls <- BIBDWheat_objects$TransferObject
  IBDWheat_rls[1:4, 2] <- c("Mean", "Blocks", "Varieties", "Plot[Block]")
  hasserls(object = BIBDWheat_objects,
           randomisation.objects = IBDWheat_rls,
           random.arrows = IBDWheat_rand_arrows,
           equation.out = "Y")

  data(Fac4Proc.dat, package = "dae")
  Fac4Proc <- Fac4Proc.dat[, -6]
  Fac4Proc_objects <- itemlist(datadesign = Fac4Proc)
  Fac4Proc_rls <- Fac4Proc_objects$TransferObject
  Fac4Proc_rls[, 2] <- Fac4Proc_rls[, 1]
  Fac4Proc_rls[16, 2] <- "Catal^Conc^Press^Temp -> Run"  
  hasserls(object = Fac4Proc_objects,
           randomisation.objects = Fac4Proc_rls,
           showpartialRLS = "N", table.out = "Y",
           smaller.fontlabelmultiplier = 2,
           hasse.font = "sans")

  data(Casuarina.dat, package = "dae")
  Casuarina <- Casuarina.dat[, -7]
  Casuarina$Row <- paste(Casuarina$Reps, Casuarina$Rows)
  Casuarina$Col <- paste(Casuarina$Reps, Casuarina$Columns)
  Casuarina <- Casuarina[, -c(2, 3)]
  Casuarina_objects <- itemlist(datadesign = Casuarina,
                                randomfacsid = c(1,0,1,0,0,0))
  Casuarina_rand_objects <- c(1:6, 9, 13)
  Casuarina_rand_arrows <- matrix(c(3, 5, 4, 13), ncol = 2, byrow = TRUE)
  Casuarina_rls <- Casuarina_objects$TransferObject
  Casuarina_rls[Casuarina_rand_objects, 2] <- c("Mean", "Countries",
                                                "InocTime", "Provences",
                                                "Reps", "Countries^InocTime",
                                                "Inoc^Provences",
                                                "{Row x Col}[Rep]")  
  hasserls(object = Casuarina_objects,
           randomisation.objects = Casuarina_rls,
           random.arrows = Casuarina_rand_arrows,
           check.confound.df = "N", showpartialRLS = "N",
           arrow.pos = 10,
           smaller.fontlabelmultiplier = 1.5,
           hasse.font = "sans")
} else {
  message("Install package 'dae' to run the final examples.")
}
}
}
\references{
Bate, S.T. and Chatfield, M.J. (2016a), Identifying the structure of the experimental design. Journal of Quality Technology, 48, 343-364.

Bate, S.T. and Chatfield, M.J. (2016b), Using the structure of the experimental design and the randomization to construct a mixed model. Journal of Quality Technology, 48, 365-387.

Box, G.E.P., Hunter, J.S., and Hunter, W.G., (1978), Statistics for Experimenters. Wiley.

Joshi, D.D. (1987), Linear Estimation and Design of Experiments. Wiley Eastern, New Delhi.

Williams, E.R., Matheson, A.C. and Harwood, C.E. (2002), Experimental design and analysis for tree improvement. 2nd edition. CSIRO, Melbourne, Australia.
}
\author{
Damianos Michaelides, Simon Bate, and Marion Chatfield
}
