/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND     
#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5Gpkg.h"      
#include "H5MMprivate.h" 

H5G_t *
H5G_rootof(H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(f);
    assert(f->shared);

    
    while (f->parent)
        f = f->parent;

    
    assert(f);
    assert(f->shared);
    assert(f->shared->root_grp);

    
    if (f->shared->root_grp->oloc.file != f)
        f->shared->root_grp->oloc.file = f;

    FUNC_LEAVE_NOAPI(f->shared->root_grp)
} 

herr_t
H5G_mkroot(H5F_t *f, bool create_root)
{
    H5G_loc_t        root_loc;               
    H5G_obj_create_t gcrt_info;              
    htri_t           stab_exists  = -1;      
    bool             sblock_dirty = false;   
    bool             path_init    = false;   
    herr_t           ret_value    = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->sblock);

    
    if (f->shared->root_grp)
        HGOTO_DONE(SUCCEED);

    
    if (H5G__node_init(f) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create group node info");

    
    if (NULL == (f->shared->root_grp = H5FL_CALLOC(H5G_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
    if (NULL == (f->shared->root_grp->shared = H5FL_CALLOC(H5G_shared_t))) {
        f->shared->root_grp = H5FL_FREE(H5G_t, f->shared->root_grp);
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
    } 

    
    root_loc.oloc = &(f->shared->root_grp->oloc);
    root_loc.path = &(f->shared->root_grp->path);
    H5G_loc_reset(&root_loc);

    
    if (create_root) {
        
        
        gcrt_info.gcpl_id    = f->shared->fcpl_id;
        gcrt_info.cache_type = H5G_NOTHING_CACHED;
        if (H5G__obj_create(f, &gcrt_info, root_loc.oloc ) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create group entry");
        if (1 != H5O_link(root_loc.oloc, 1))
            HGOTO_ERROR(H5E_SYM, H5E_LINKCOUNT, FAIL, "internal error (wrong link count)");

        
        if (H5O_dec_rc_by_loc(root_loc.oloc) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTDEC, FAIL,
                        "unable to decrement refcount on root group's object header");

        
        sblock_dirty = true;

        
        assert(!f->shared->sblock->root_ent);
        if (f->shared->sblock->super_vers < HDF5_SUPERBLOCK_VERSION_2) {
            
            if (NULL == (f->shared->sblock->root_ent = (H5G_entry_t *)H5MM_calloc(sizeof(H5G_entry_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "can't allocate space for symbol table entry");

            
            f->shared->sblock->root_ent->type = gcrt_info.cache_type;
            if (gcrt_info.cache_type != H5G_NOTHING_CACHED)
                f->shared->sblock->root_ent->cache = gcrt_info.cache;
            f->shared->sblock->root_ent->name_off = 0; 
            f->shared->sblock->root_ent->header   = root_loc.oloc->addr;
        } 
    }     
    else {
        
        root_loc.oloc->addr = f->shared->sblock->root_addr;
        root_loc.oloc->file = f;

        
        if (H5O_open(root_loc.oloc) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open root group");

        
        if (f->shared->sblock->root_ent && f->shared->sblock->root_ent->type == H5G_CACHED_STAB) {
            
            if ((stab_exists = H5O_msg_exists(root_loc.oloc, H5O_STAB_ID)) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check if symbol table message exists");

            
            if (!stab_exists)
                f->shared->sblock->root_ent->type = H5G_NOTHING_CACHED;
#ifndef H5_STRICT_FORMAT_CHECKS
            
            else if (H5F_INTENT(f) & H5F_ACC_RDWR) {
                H5O_stab_t cached_stab;

                
                cached_stab.btree_addr = f->shared->sblock->root_ent->cache.stab.btree_addr;
                cached_stab.heap_addr  = f->shared->sblock->root_ent->cache.stab.heap_addr;

                
                if (H5G__stab_valid(root_loc.oloc, &cached_stab) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to verify symbol table");
            } 
#endif        
        }     
    }         

    
    if ((H5F_INTENT(f) & H5F_ACC_RDWR) && stab_exists != false && f->shared->sblock->root_ent &&
        f->shared->sblock->root_ent->type != H5G_CACHED_STAB) {
        H5O_stab_t stab; 

        
        if (stab_exists == -1 && (stab_exists = H5O_msg_exists(root_loc.oloc, H5O_STAB_ID)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check if symbol table message exists");

        if (stab_exists) {
            
            if (NULL == H5O_msg_read(root_loc.oloc, H5O_STAB_ID, &stab))
                HGOTO_ERROR(H5E_SYM, H5E_BADMESG, FAIL, "unable to read symbol table message");

            
            f->shared->sblock->root_ent->type                  = H5G_CACHED_STAB;
            f->shared->sblock->root_ent->cache.stab.btree_addr = stab.btree_addr;
            f->shared->sblock->root_ent->cache.stab.heap_addr  = stab.heap_addr;

            
            sblock_dirty = true;
        } 
    }     

    
    H5G__name_init(root_loc.path, "/");
    path_init = true;

    f->shared->root_grp->shared->fo_count = 1;
    
    assert((1 == f->nopen_objs) || (2 == f->nopen_objs && HADDR_UNDEF != f->shared->sblock->ext_addr));
    f->nopen_objs--;

done:
    
    if (ret_value < 0) {
        if (f->shared->root_grp) {
            if (path_init)
                H5G_name_free(root_loc.path);
            if (f->shared->root_grp->shared)
                f->shared->root_grp->shared = H5FL_FREE(H5G_shared_t, f->shared->root_grp->shared);
            f->shared->root_grp = H5FL_FREE(H5G_t, f->shared->root_grp);
        } 
        if (f->shared->sblock)
            f->shared->sblock->root_ent = (H5G_entry_t *)H5MM_xfree(f->shared->sblock->root_ent);
    } 

    
    if (sblock_dirty)
        if (H5AC_mark_entry_dirty(f->shared->sblock) < 0)
            HDONE_ERROR(H5E_FILE, H5E_CANTMARKDIRTY, FAIL, "unable to mark superblock as dirty");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_root_free(H5G_t *grp)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(grp && grp->shared);
    assert(grp->shared->fo_count > 0);

    
    H5G_name_free(&(grp->path));

    grp->shared = H5FL_FREE(H5G_shared_t, grp->shared);
    grp         = H5FL_FREE(H5G_t, grp);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5G_root_loc(H5F_t *f, H5G_loc_t *loc)
{
    H5G_t *root_grp;            
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(f);
    assert(loc);

    
    root_grp = H5G_rootof(f);
    assert(root_grp);

    
    if (NULL == (loc->oloc = H5G_oloc(root_grp)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get object location for root group");
    if (NULL == (loc->path = H5G_nameof(root_grp)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "unable to get path for root group");

    
    
    
    if (!H5F_is_mount(f)) {
        loc->oloc->file         = f;
        loc->oloc->holding_file = false;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
